'use client';

import { useState, useEffect, useRef } from 'react';
import { Database, Globe, Server, Network, Search, Menu, X, Plus, Edit, Trash2, Activity, Download, Upload, Filter, Shield, Cloud, Globe2, ExternalLink, AlertTriangle, CheckCircle, Clock, Users, Cpu, HardDrive, MapPin, Mail, Lock, Eye, EyeOff, Copy, RefreshCw, FileText, Database as DatabaseIcon, Settings, BarChart3, TrendingUp, Calendar, Tag } from 'lucide-react';

// Enhanced interfaces
interface Website {
  id: number;
  name: string;
  domain: string;
  ip_address: string;
  description?: string;
  status: 'active' | 'inactive' | 'suspended' | 'redirected';
  redirect_url?: string;
  created_at: string;
  updated_at?: string;
  tags?: string[];
}

interface Host {
  id: number;
  name: string;
  ip_address: string;
  provider?: string;
  location?: string;
  specs?: string;
  status: 'active' | 'inactive' | 'maintenance';
  created_at: string;
  updated_at?: string;
  tags?: string[];
}

interface DnsRecord {
  id: number;
  domain: string;
  type: 'A' | 'AAAA' | 'CNAME' | 'MX' | 'TXT' | 'NS' | 'SRV';
  value: string;
  cloudflare_account: string;
  email: string;
  password?: string;
  ttl?: number;
  proxy_status: boolean;
  created_at: string;
}

interface FilterOptions {
  status: string[];
  ip_address: string;
  provider: string;
  date_range: string;
  tags: string[];
}

interface DashboardStats {
  total_websites: number;
  active_websites: number;
  redirected_websites: number;
  total_hosts: number;
  active_hosts: number;
  unique_ips: number;
  total_dns_records: number;
  cloudflare_accounts: number;
}

export default function Home() {
  const [currentPage, setCurrentPage] = useState('dashboard');
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const [loading, setLoading] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');
  const [stats, setStats] = useState<DashboardStats | null>(null);
  const [websites, setWebsites] = useState<Website[]>([]);
  const [hosts, setHosts] = useState<Host[]>([]);
  const [dnsRecords, setDnsRecords] = useState<DnsRecord[]>([]);
  const [ipDistribution, setIpDistribution] = useState<any[]>([]);
  const [showWebsiteModal, setShowWebsiteModal] = useState(false);
  const [showHostModal, setShowHostModal] = useState(false);
  const [showDnsModal, setShowDnsModal] = useState(false);
  const [showBatchImportModal, setShowBatchImportModal] = useState(false);
  const [showFilterModal, setShowFilterModal] = useState(false);
  const [editingWebsite, setEditingWebsite] = useState<Website | null>(null);
  const [editingHost, setEditingHost] = useState<Host | null>(null);
  const [editingDns, setEditingDns] = useState<DnsRecord | null>(null);
  const [toast, setToast] = useState<{ message: string; type: 'success' | 'error' | 'warning' } | null>(null);
  const [showPassword, setShowPassword] = useState<{ [key: string]: boolean }>({});
  const [filters, setFilters] = useState<FilterOptions>({
    status: [],
    ip_address: '',
    provider: '',
    date_range: '',
    tags: []
  });
  const [batchImportData, setBatchImportData] = useState('');
  const [importType, setImportType] = useState<'websites' | 'hosts'>('websites');
  
  const fileInputRef = useRef<HTMLInputElement>(null);

  // Enhanced form states
  const [websiteForm, setWebsiteForm] = useState({
    name: '',
    domain: '',
    ip_address: '',
    description: '',
    status: 'active' as 'active' | 'inactive' | 'suspended' | 'redirected',
    redirect_url: '',
    tags: [] as string[]
  });

  const [hostForm, setHostForm] = useState({
    name: '',
    ip_address: '',
    provider: '',
    location: '',
    specs: '',
    status: 'active' as 'active' | 'inactive' | 'maintenance',
    tags: [] as string[]
  });

  const [dnsForm, setDnsForm] = useState({
    domain: '',
    type: 'A' as 'A' | 'AAAA' | 'CNAME' | 'MX' | 'TXT' | 'NS' | 'SRV',
    value: '',
    cloudflare_account: '',
    email: '',
    password: '',
    ttl: 3600,
    proxy_status: false
  });

  useEffect(() => {
    loadDashboard();
  }, []);

  const loadDashboard = async () => {
    try {
      setLoading(true);
      const response = await fetch('/api/dashboard');
      const data = await response.json();
      
      if (data.success) {
        setStats(data.data.stats);
        setIpDistribution(data.data.ipDistribution);
      }
    } catch (error) {
      console.error('Failed to load dashboard:', error);
      showToast('خطا در بارگذاری داشبورد', 'error');
    } finally {
      setLoading(false);
    }
  };

  const loadWebsites = async () => {
    try {
      setLoading(true);
      const response = await fetch('/api/websites');
      const data = await response.json();
      
      if (data.success) {
        setWebsites(data.data);
      }
    } catch (error) {
      console.error('Failed to load websites:', error);
      showToast('خطا در بارگذاری سایت‌ها', 'error');
    } finally {
      setLoading(false);
    }
  };

  const loadHosts = async () => {
    try {
      setLoading(true);
      const response = await fetch('/api/hosts');
      const data = await response.json();
      
      if (data.success) {
        setHosts(data.data);
      }
    } catch (error) {
      console.error('Failed to load hosts:', error);
      showToast('خطا در بارگذاری هاست‌ها', 'error');
    } finally {
      setLoading(false);
    }
  };

  const loadDnsRecords = async () => {
    try {
      setLoading(true);
      const response = await fetch('/api/dns');
      const data = await response.json();
      
      if (data.success) {
        setDnsRecords(data.data);
      }
    } catch (error) {
      console.error('Failed to load DNS records:', error);
      showToast('خطا در بارگذاری رکوردهای DNS', 'error');
    } finally {
      setLoading(false);
    }
  };

  const handlePageChange = (page: string) => {
    setCurrentPage(page);
    setSidebarOpen(false);
    
    switch (page) {
      case 'dashboard':
        loadDashboard();
        break;
      case 'websites':
        loadWebsites();
        break;
      case 'hosts':
        loadHosts();
        break;
      case 'dns':
        loadDnsRecords();
        break;
    }
  };

  const showToast = (message: string, type: 'success' | 'error' | 'warning' = 'success') => {
    setToast({ message, type });
    setTimeout(() => setToast(null), 3000);
  };

  // CSV Export functionality
  const exportToCSV = (type: 'websites' | 'hosts' | 'dns') => {
    let data: any[] = [];
    let filename = '';
    let headers: string[] = [];

    switch (type) {
      case 'websites':
        data = websites;
        filename = 'websites.csv';
        headers = ['ID', 'Name', 'Domain', 'IP Address', 'Status', 'Redirect URL', 'Description', 'Created At'];
        break;
      case 'hosts':
        data = hosts;
        filename = 'hosts.csv';
        headers = ['ID', 'Name', 'IP Address', 'Provider', 'Location', 'Status', 'Specs', 'Created At'];
        break;
      case 'dns':
        data = dnsRecords;
        filename = 'dns-records.csv';
        headers = ['ID', 'Domain', 'Type', 'Value', 'Cloudflare Account', 'Email', 'Proxy Status', 'Created At'];
        break;
    }

    const csvContent = [
      headers.join(','),
      ...data.map(item => {
        switch (type) {
          case 'websites':
            return `${item.id},"${item.name}","${item.domain}","${item.ip_address}","${item.status}","${item.redirect_url || ''}","${item.description || ''}","${item.created_at}"`;
          case 'hosts':
            return `${item.id},"${item.name}","${item.ip_address}","${item.provider || ''}","${item.location || ''}","${item.status}","${item.specs || ''}","${item.created_at}"`;
          case 'dns':
            return `${item.id},"${item.domain}","${item.type}","${item.value}","${item.cloudflare_account}","${item.email}","${item.proxy_status}","${item.created_at}"`;
          default:
            return '';
        }
      })
    ].join('\n');

    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', filename);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    
    showToast(`فایل ${filename} با موفقیت دانلود شد`, 'success');
  };

  // Batch import functionality
  const handleBatchImport = () => {
    try {
      const lines = batchImportData.trim().split('\n');
      const results: any[] = [];
      
      lines.forEach((line, index) => {
        if (importType === 'websites') {
          // Expected format: name,domain,ip_address,description
          const [name, domain, ip_address, description] = line.split(',').map(item => item.trim());
          if (name && domain && ip_address) {
            results.push({
              name,
              domain,
              ip_address,
              description: description || '',
              status: 'active'
            });
          }
        } else if (importType === 'hosts') {
          // Expected format: name,ip_address,provider,location,specs
          const [name, ip_address, provider, location, specs] = line.split(',').map(item => item.trim());
          if (name && ip_address) {
            results.push({
              name,
              ip_address,
              provider: provider || '',
              location: location || '',
              specs: specs || '',
              status: 'active'
            });
          }
        }
      });

      if (results.length > 0) {
        // Here you would normally send this to the API
        showToast(`${results.length} رکورد با موفقیت برای وارد شدن آماده شد`, 'success');
        setBatchImportData('');
        setShowBatchImportModal(false);
        
        // Reload the current page data
        if (importType === 'websites') {
          loadWebsites();
        } else {
          loadHosts();
        }
      } else {
        showToast('هیچ رکورد معتبری یافت نشد', 'error');
      }
    } catch (error) {
      console.error('Import error:', error);
      showToast('خطا در وارد کردن داده‌ها', 'error');
    }
  };

  // File upload for CSV import
  const handleFileUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      const reader = new FileReader();
      reader.onload = (e) => {
        const content = e.target?.result as string;
        setBatchImportData(content);
      };
      reader.readAsText(file);
    }
  };

  // Filter functionality
  const applyFilters = () => {
    // Apply filters to current data
    let filteredWebsites = [...websites];
    let filteredHosts = [...hosts];

    if (filters.status.length > 0) {
      filteredWebsites = filteredWebsites.filter(w => filters.status.includes(w.status));
      filteredHosts = filteredHosts.filter(h => filters.status.includes(h.status));
    }

    if (filters.ip_address) {
      filteredWebsites = filteredWebsites.filter(w => w.ip_address.includes(filters.ip_address));
      filteredHosts = filteredHosts.filter(h => h.ip_address.includes(filters.ip_address));
    }

    if (filters.provider) {
      filteredHosts = filteredHosts.filter(h => h.provider?.includes(filters.provider));
    }

    setWebsites(filteredWebsites);
    setHosts(filteredHosts);
    setShowFilterModal(false);
    showToast('فیلترها با موفقیت اعمال شد', 'success');
  };

  const clearFilters = () => {
    setFilters({
      status: [],
      ip_address: '',
      provider: '',
      date_range: '',
      tags: []
    });
    loadWebsites();
    loadHosts();
    showToast('فیلترها پاک شدند', 'success');
  };

  // Copy to clipboard functionality
  const copyToClipboard = (text: string) => {
    navigator.clipboard.writeText(text).then(() => {
      showToast('متن با موفقیت کپی شد', 'success');
    });
  };

  // Enhanced form handlers
  const resetWebsiteForm = () => {
    setWebsiteForm({
      name: '',
      domain: '',
      ip_address: '',
      description: '',
      status: 'active',
      redirect_url: '',
      tags: []
    });
    setEditingWebsite(null);
  };

  const resetHostForm = () => {
    setHostForm({
      name: '',
      ip_address: '',
      provider: '',
      location: '',
      specs: '',
      status: 'active',
      tags: []
    });
    setEditingHost(null);
  };

  const resetDnsForm = () => {
    setDnsForm({
      domain: '',
      type: 'A',
      value: '',
      cloudflare_account: '',
      email: '',
      password: '',
      ttl: 3600,
      proxy_status: false
    });
    setEditingDns(null);
  };

  const openWebsiteModal = (website?: Website) => {
    if (website) {
      setEditingWebsite(website);
      setWebsiteForm({
        name: website.name,
        domain: website.domain,
        ip_address: website.ip_address,
        description: website.description || '',
        status: website.status,
        redirect_url: website.redirect_url || '',
        tags: website.tags || []
      });
    } else {
      resetWebsiteForm();
    }
    setShowWebsiteModal(true);
  };

  const openHostModal = (host?: Host) => {
    if (host) {
      setEditingHost(host);
      setHostForm({
        name: host.name,
        ip_address: host.ip_address,
        provider: host.provider || '',
        location: host.location || '',
        specs: host.specs || '',
        status: host.status,
        tags: host.tags || []
      });
    } else {
      resetHostForm();
    }
    setShowHostModal(true);
  };

  const openDnsModal = (dns?: DnsRecord) => {
    if (dns) {
      setEditingDns(dns);
      setDnsForm({
        domain: dns.domain,
        type: dns.type,
        value: dns.value,
        cloudflare_account: dns.cloudflare_account,
        email: dns.email,
        password: dns.password || '',
        ttl: dns.ttl || 3600,
        proxy_status: dns.proxy_status
      });
    } else {
      resetDnsForm();
    }
    setShowDnsModal(true);
  };

  const saveWebsite = async () => {
    try {
      const url = editingWebsite ? `/api/websites?id=${editingWebsite.id}` : '/api/websites';
      const method = editingWebsite ? 'PUT' : 'POST';
      
      const response = await fetch(url, {
        method,
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(websiteForm)
      });
      
      const data = await response.json();
      
      if (data.success) {
        showToast(editingWebsite ? 'سایت با موفقیت ویرایش شد' : 'سایت با موفقیت افزوده شد', 'success');
        setShowWebsiteModal(false);
        resetWebsiteForm();
        loadWebsites();
      } else {
        showToast(data.message || 'خطا در ذخیره سایت', 'error');
      }
    } catch (error) {
      console.error('Failed to save website:', error);
      showToast('خطا در ذخیره سایت', 'error');
    }
  };

  const saveHost = async () => {
    try {
      const url = editingHost ? `/api/hosts?id=${editingHost.id}` : '/api/hosts';
      const method = editingHost ? 'PUT' : 'POST';
      
      const response = await fetch(url, {
        method,
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(hostForm)
      });
      
      const data = await response.json();
      
      if (data.success) {
        showToast(editingHost ? 'هاست با موفقیت ویرایش شد' : 'هاست با موفقیت افزوده شد', 'success');
        setShowHostModal(false);
        resetHostForm();
        loadHosts();
      } else {
        showToast(data.message || 'خطا در ذخیره هاست', 'error');
      }
    } catch (error) {
      console.error('Failed to save host:', error);
      showToast('خطا در ذخیره هاست', 'error');
    }
  };

  const saveDns = async () => {
    try {
      const url = editingDns ? `/api/dns?id=${editingDns.id}` : '/api/dns';
      const method = editingDns ? 'PUT' : 'POST';
      
      const response = await fetch(url, {
        method,
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(dnsForm)
      });
      
      const data = await response.json();
      
      if (data.success) {
        showToast(editingDns ? 'رکورد DNS با موفقیت ویرایش شد' : 'رکورد DNS با موفقیت افزوده شد', 'success');
        setShowDnsModal(false);
        resetDnsForm();
        loadDnsRecords();
      } else {
        showToast(data.message || 'خطا در ذخیره رکورد DNS', 'error');
      }
    } catch (error) {
      console.error('Failed to save DNS record:', error);
      showToast('خطا در ذخیره رکورد DNS', 'error');
    }
  };

  const deleteWebsite = async (id: number) => {
    if (!confirm('آیا از حذف این سایت اطمینان دارید؟')) return;
    
    try {
      const response = await fetch(`/api/websites?id=${id}`, { method: 'DELETE' });
      const data = await response.json();
      
      if (data.success) {
        showToast('سایت با موفقیت حذف شد', 'success');
        loadWebsites();
      } else {
        showToast(data.message || 'خطا در حذف سایت', 'error');
      }
    } catch (error) {
      console.error('Failed to delete website:', error);
      showToast('خطا در حذف سایت', 'error');
    }
  };

  const deleteHost = async (id: number) => {
    if (!confirm('آیا از حذف این هاست اطمینان دارید؟')) return;
    
    try {
      const response = await fetch(`/api/hosts?id=${id}`, { method: 'DELETE' });
      const data = await response.json();
      
      if (data.success) {
        showToast('هاست با موفقیت حذف شد', 'success');
        loadHosts();
      } else {
        showToast(data.message || 'خطا در حذف هاست', 'error');
      }
    } catch (error) {
      console.error('Failed to delete host:', error);
      showToast('خطا در حذف هاست', 'error');
    }
  };

  const deleteDns = async (id: number) => {
    if (!confirm('آیا از حذف این رکورد DNS اطمینان دارید؟')) return;
    
    try {
      const response = await fetch(`/api/dns?id=${id}`, { method: 'DELETE' });
      const data = await response.json();
      
      if (data.success) {
        showToast('رکورد DNS با موفقیت حذف شد', 'success');
        loadDnsRecords();
      } else {
        showToast(data.message || 'خطا در حذف رکورد DNS', 'error');
      }
    } catch (error) {
      console.error('Failed to delete DNS record:', error);
      showToast('خطا در حذف رکورد DNS', 'error');
    }
  };

  const getStatusBadge = (status: string) => {
    const styles = {
      active: 'bg-green-100 text-green-800',
      inactive: 'bg-red-100 text-red-800',
      suspended: 'bg-orange-100 text-orange-800',
      maintenance: 'bg-yellow-100 text-yellow-800',
      redirected: 'bg-blue-100 text-blue-800'
    };
    
    const labels = {
      active: 'فعال',
      inactive: 'غیرفعال',
      suspended: 'مسدود',
      maintenance: 'در حال تعمیر',
      redirected: 'ریدایرکت شده'
    };
    
    return (
      <span className={`px-2 py-1 rounded-full text-xs font-medium ${styles[status as keyof typeof styles]}`}>
        {labels[status as keyof typeof labels]}
      </span>
    );
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('fa-IR', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  const menuItems = [
    { id: 'dashboard', label: 'داشبورد', icon: BarChart3 },
    { id: 'websites', label: 'سایت‌ها', icon: Globe },
    { id: 'hosts', label: 'هاست‌ها', icon: Server },
    { id: 'dns', label: 'مدیریت DNS', icon: Cloud },
    { id: 'ip-management', label: 'مدیریت IP', icon: Network },
    { id: 'security', label: 'امنیت', icon: Shield },
  ];

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-purple-50" dir="rtl" style={{ fontFamily: 'Vazirmatn, sans-serif' }}>
      {/* Enhanced Toast Notification */}
      {toast && (
        <div className={`fixed top-4 left-1/2 transform -translate-x-1/2 z-50 px-6 py-3 rounded-lg shadow-lg flex items-center space-x-reverse space-x-2 ${
          toast.type === 'success' ? 'bg-green-500' : 
          toast.type === 'warning' ? 'bg-yellow-500' : 'bg-red-500'
        } text-white font-medium`}>
          {toast.type === 'success' && <CheckCircle size={20} />}
          {toast.type === 'warning' && <AlertTriangle size={20} />}
          {toast.type === 'error' && <X size={20} />}
          <span>{toast.message}</span>
        </div>
      )}

      {/* Enhanced Sidebar */}
      <div className={`fixed top-0 right-0 h-64 w-64 bg-gray-900 text-white z-40 transform transition-transform duration-300 ${
        sidebarOpen ? 'translate-x-0' : 'translate-x-full'
      } md:translate-x-0 md:h-screen`}>
        <div className="p-6 border-b border-gray-700">
          <h2 className="text-xl font-bold flex items-center">
            <Database className="ml-2" />
            سیستم بایگانی
          </h2>
          <p className="text-gray-400 text-sm mt-2">نسخه پیشرفته 2.0</p>
        </div>
        <nav>
          {menuItems.map((item) => {
            const Icon = item.icon;
            return (
              <div
                key={item.id}
                onClick={() => handlePageChange(item.id)}
                className={`flex items-center px-6 py-4 cursor-pointer transition-colors ${
                  currentPage === item.id
                    ? 'bg-blue-600 border-r-4 border-blue-400'
                    : 'hover:bg-gray-800'
                }`}
              >
                <Icon className="ml-3" size={20} />
                {item.label}
              </div>
            );
          })}
        </nav>
      </div>

      {/* Main Content */}
      <div className="md:mr-64">
        {/* Enhanced Header */}
        <header className="bg-white shadow-sm border-b">
          <div className="px-4 py-4 flex justify-between items-center">
            <div className="flex items-center">
              <button
                onClick={() => setSidebarOpen(!sidebarOpen)}
                className="md:hidden p-2 rounded-lg hover:bg-gray-100"
              >
                {sidebarOpen ? <X size={24} /> : <Menu size={24} />}
              </button>
              <div className="mr-4">
                <h1 className="text-2xl font-bold text-gray-800">سیستم مدیریت بایگانی پیشرفته</h1>
                <p className="text-gray-600 text-sm">مدیریت سایت‌ها، هاست‌ها، DNS و امنیت بر اساس آدرس IP</p>
              </div>
            </div>
            <div className="flex items-center space-x-reverse space-x-3">
              <div className="relative">
                <Search className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400" size={20} />
                <input
                  type="text"
                  placeholder="جستجوی پیشرفته..."
                  className="pl-10 pr-10 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                />
              </div>
              <button
                onClick={() => setShowFilterModal(true)}
                className="p-2 bg-gray-100 rounded-lg hover:bg-gray-200 transition-colors"
                title="فیلتر پیشرفته"
              >
                <Filter size={20} />
              </button>
              <button
                onClick={() => window.location.reload()}
                className="p-2 bg-gray-100 rounded-lg hover:bg-gray-200 transition-colors"
                title="تازه‌سازی"
              >
                <RefreshCw size={20} />
              </button>
            </div>
          </div>
        </header>

        {/* Content */}
        <main className="p-6">
          {loading && (
            <div className="flex justify-center items-center h-64">
              <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
            </div>
          )}

          {/* Dashboard Page */}
          {currentPage === 'dashboard' && !loading && (
            <div>
              {/* Enhanced Stats Cards */}
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                <div className="bg-gradient-to-r from-blue-500 to-blue-600 text-white p-6 rounded-xl shadow-lg">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-blue-100">کل سایت‌ها</p>
                      <p className="text-3xl font-bold mt-2">{stats?.total_websites || 0}</p>
                      <p className="text-blue-200 text-sm mt-1">{stats?.redirected_websites || 0} ریدایرکت شده</p>
                    </div>
                    <Globe size={40} className="text-blue-200" />
                  </div>
                </div>
                <div className="bg-gradient-to-r from-green-500 to-green-600 text-white p-6 rounded-xl shadow-lg">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-green-100">سایت‌های فعال</p>
                      <p className="text-3xl font-bold mt-2">{stats?.active_websites || 0}</p>
                    </div>
                    <div className="w-10 h-10 bg-green-200 rounded-full flex items-center justify-center">
                      <CheckCircle className="text-green-600" size={24} />
                    </div>
                  </div>
                </div>
                <div className="bg-gradient-to-r from-purple-500 to-purple-600 text-white p-6 rounded-xl shadow-lg">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-purple-100">کل هاست‌ها</p>
                      <p className="text-3xl font-bold mt-2">{stats?.total_hosts || 0}</p>
                    </div>
                    <Server size={40} className="text-purple-200" />
                  </div>
                </div>
                <div className="bg-gradient-to-r from-orange-500 to-orange-600 text-white p-6 rounded-xl shadow-lg">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-orange-100">IP های منحصر به فرد</p>
                      <p className="text-3xl font-bold mt-2">{stats?.unique_ips || 0}</p>
                    </div>
                    <Network size={40} className="text-orange-200" />
                  </div>
                </div>
              </div>

              {/* Additional Stats Row */}
              <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
                <div className="bg-gradient-to-r from-cyan-500 to-cyan-600 text-white p-6 rounded-xl shadow-lg">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-cyan-100">رکوردهای DNS</p>
                      <p className="text-2xl font-bold mt-2">{stats?.total_dns_records || 0}</p>
                    </div>
                    <Cloud size={32} className="text-cyan-200" />
                  </div>
                </div>
                <div className="bg-gradient-to-r from-indigo-500 to-indigo-600 text-white p-6 rounded-xl shadow-lg">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-indigo-100">اکانت‌های Cloudflare</p>
                      <p className="text-2xl font-bold mt-2">{stats?.cloudflare_accounts || 0}</p>
                    </div>
                    <Shield size={32} className="text-indigo-200" />
                  </div>
                </div>
                <div className="bg-gradient-to-r from-pink-500 to-pink-600 text-white p-6 rounded-xl shadow-lg">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-pink-100">هاست‌های فعال</p>
                      <p className="text-2xl font-bold mt-2">{stats?.active_hosts || 0}</p>
                    </div>
                    <Activity size={32} className="text-pink-200" />
                  </div>
                </div>
              </div>

              {/* IP Distribution */}
              <div className="bg-white rounded-xl shadow-lg p-6 mb-6">
                <div className="flex justify-between items-center mb-4">
                  <h3 className="text-xl font-bold text-gray-800 flex items-center">
                    <Network className="ml-2" />
                    توزیع IP ها
                  </h3>
                  <button
                    onClick={() => exportToCSV('websites')}
                    className="flex items-center px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors"
                  >
                    <Download size={16} className="ml-2" />
                    export CSV
                  </button>
                </div>
                <div className="space-y-4">
                  {ipDistribution.length > 0 ? (
                    ipDistribution.map((ip) => (
                      <div key={ip.ip_address} className="border rounded-lg p-4 hover:shadow-md transition-shadow">
                        <div className="flex justify-between items-center">
                          <div>
                            <h4 className="font-bold text-lg flex items-center">
                              {ip.ip_address}
                              <button
                                onClick={() => copyToClipboard(ip.ip_address)}
                                className="mr-2 p-1 hover:bg-gray-100 rounded"
                                title="کپی آدرس IP"
                              >
                                <Copy size={16} />
                              </button>
                            </h4>
                            <div className="flex space-x-reverse space-x-4 mt-2 text-sm text-gray-600">
                              <span className="flex items-center">
                                <Globe size={16} className="ml-1" />
                                {ip.total_websites} سایت
                              </span>
                              <span className="flex items-center">
                                <Server size={16} className="ml-1" />
                                {ip.total_hosts} هاست
                              </span>
                            </div>
                          </div>
                          <div className="text-left">
                            <div className="text-2xl font-bold text-blue-600">{ip.total_items}</div>
                            <div className="text-xs text-gray-500">مجموع</div>
                          </div>
                        </div>
                        <div className="mt-3">
                          <div className="w-full bg-gray-200 rounded-full h-2">
                            <div
                              className="bg-gradient-to-r from-blue-500 to-purple-600 h-2 rounded-full"
                              style={{
                                width: `${(ip.total_items / Math.max(...ipDistribution.map(item => item.total_items))) * 100}%`
                              }}
                            ></div>
                          </div>
                        </div>
                      </div>
                    ))
                  ) : (
                    <p className="text-gray-500 text-center py-8">هیچ IP یافت نشد</p>
                  )}
                </div>
              </div>
            </div>
          )}

          {/* Websites Page */}
          {currentPage === 'websites' && !loading && (
            <div className="bg-white rounded-xl shadow-lg p-6">
              <div className="flex justify-between items-center mb-6">
                <h2 className="text-2xl font-bold text-gray-800 flex items-center">
                  <Globe className="ml-2" />
                  مدیریت سایت‌ها
                </h2>
                <div className="flex space-x-reverse space-x-3">
                  <button
                    onClick={() => setShowBatchImportModal(true)}
                    className="flex items-center px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700 transition-colors"
                  >
                    <Upload size={16} className="ml-2" />
                    وارد کردن دسته جمعی
                  </button>
                  <button
                    onClick={() => exportToCSV('websites')}
                    className="flex items-center px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors"
                  >
                    <Download size={16} className="ml-2" />
                    export CSV
                  </button>
                  <button
                    onClick={() => openWebsiteModal()}
                    className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                  >
                    <Plus size={16} className="ml-2" />
                    افزودن سایت
                  </button>
                </div>
              </div>
              <div className="space-y-4">
                {websites.length > 0 ? (
                  websites.map((website) => (
                    <div key={website.id} className="border rounded-lg p-4 hover:shadow-md transition-shadow">
                      <div className="flex justify-between items-start">
                        <div className="flex-1">
                          <div className="flex items-center">
                            <h4 className="font-bold text-lg">{website.name}</h4>
                            {website.status === 'redirected' && (
                              <ExternalLink size={16} className="mr-2 text-blue-600" title="ریدایرکت شده" />
                            )}
                          </div>
                          <p className="text-gray-600">{website.domain}</p>
                          <div className="flex items-center mt-2 text-sm text-gray-500">
                            <Network size={16} className="ml-2" />
                            <span>{website.ip_address}</span>
                            <button
                              onClick={() => copyToClipboard(website.ip_address)}
                              className="mr-2 p-1 hover:bg-gray-100 rounded"
                              title="کپی IP"
                            >
                              <Copy size={12} />
                            </button>
                            <span className="mx-2">•</span>
                            <span>{formatDate(website.created_at)}</span>
                          </div>
                          {website.redirect_url && (
                            <div className="mt-2 p-2 bg-blue-50 rounded text-sm">
                              <span className="text-blue-800 font-medium">ریدایرکت به:</span>
                              <a 
                                href={website.redirect_url} 
                                target="_blank" 
                                rel="noopener noreferrer"
                                className="mr-2 text-blue-600 hover:underline"
                              >
                                {website.redirect_url}
                              </a>
                            </div>
                          )}
                          {website.description && (
                            <p className="text-gray-600 mt-2">{website.description}</p>
                          )}
                          {website.tags && website.tags.length > 0 && (
                            <div className="flex flex-wrap gap-2 mt-2">
                              {website.tags.map((tag, index) => (
                                <span key={index} className="px-2 py-1 bg-gray-100 text-gray-600 rounded-full text-xs">
                                  {tag}
                                </span>
                              ))}
                            </div>
                          )}
                        </div>
                        <div className="flex items-center space-x-reverse space-x-2">
                          {getStatusBadge(website.status)}
                          <button
                            onClick={() => openWebsiteModal(website)}
                            className="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
                          >
                            <Edit size={18} />
                          </button>
                          <button
                            onClick={() => deleteWebsite(website.id)}
                            className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors"
                          >
                            <Trash2 size={18} />
                          </button>
                        </div>
                      </div>
                    </div>
                  ))
                ) : (
                  <p className="text-gray-500 text-center py-8">هیچ سایتی یافت نشد</p>
                )}
              </div>
            </div>
          )}

          {/* Hosts Page */}
          {currentPage === 'hosts' && !loading && (
            <div className="bg-white rounded-xl shadow-lg p-6">
              <div className="flex justify-between items-center mb-6">
                <h2 className="text-2xl font-bold text-gray-800 flex items-center">
                  <Server className="ml-2" />
                  مدیریت هاست‌ها
                </h2>
                <div className="flex space-x-reverse space-x-3">
                  <button
                    onClick={() => setShowBatchImportModal(true)}
                    className="flex items-center px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700 transition-colors"
                  >
                    <Upload size={16} className="ml-2" />
                    وارد کردن دسته جمعی
                  </button>
                  <button
                    onClick={() => exportToCSV('hosts')}
                    className="flex items-center px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors"
                  >
                    <Download size={16} className="ml-2" />
                    export CSV
                  </button>
                  <button
                    onClick={() => openHostModal()}
                    className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                  >
                    <Plus size={16} className="ml-2" />
                    افزودن هاست
                  </button>
                </div>
              </div>
              <div className="space-y-4">
                {hosts.length > 0 ? (
                  hosts.map((host) => (
                    <div key={host.id} className="border rounded-lg p-4 hover:shadow-md transition-shadow">
                      <div className="flex justify-between items-start">
                        <div className="flex-1">
                          <h4 className="font-bold text-lg">{host.name}</h4>
                          <div className="flex items-center mt-2 text-sm text-gray-500">
                            <Network size={16} className="ml-2" />
                            <span>{host.ip_address}</span>
                            <button
                              onClick={() => copyToClipboard(host.ip_address)}
                              className="mr-2 p-1 hover:bg-gray-100 rounded"
                              title="کپی IP"
                            >
                              <Copy size={12} />
                            </button>
                            {host.provider && (
                              <>
                                <span className="mx-2">•</span>
                                <span>{host.provider}</span>
                              </>
                            )}
                            {host.location && (
                              <>
                                <span className="mx-2">•</span>
                                <MapPin size={14} className="ml-1" />
                                <span>{host.location}</span>
                              </>
                            )}
                          </div>
                          <div className="text-xs text-gray-500 mt-1">{formatDate(host.created_at)}</div>
                          {host.specs && (
                            <div className="mt-2 p-2 bg-gray-50 rounded text-sm">
                              <div className="flex items-center text-gray-700">
                                <Cpu size={14} className="ml-1" />
                                <span>{host.specs}</span>
                              </div>
                            </div>
                          )}
                          {host.tags && host.tags.length > 0 && (
                            <div className="flex flex-wrap gap-2 mt-2">
                              {host.tags.map((tag, index) => (
                                <span key={index} className="px-2 py-1 bg-gray-100 text-gray-600 rounded-full text-xs">
                                  {tag}
                                </span>
                              ))}
                            </div>
                          )}
                        </div>
                        <div className="flex items-center space-x-reverse space-x-2">
                          {getStatusBadge(host.status)}
                          <button
                            onClick={() => openHostModal(host)}
                            className="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
                          >
                            <Edit size={18} />
                          </button>
                          <button
                            onClick={() => deleteHost(host.id)}
                            className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors"
                          >
                            <Trash2 size={18} />
                          </button>
                        </div>
                      </div>
                    </div>
                  ))
                ) : (
                  <p className="text-gray-500 text-center py-8">هیچ هاستی یافت نشد</p>
                )}
              </div>
            </div>
          )}

          {/* DNS Management Page */}
          {currentPage === 'dns' && !loading && (
            <div className="bg-white rounded-xl shadow-lg p-6">
              <div className="flex justify-between items-center mb-6">
                <h2 className="text-2xl font-bold text-gray-800 flex items-center">
                  <Cloud className="ml-2" />
                  مدیریت DNS
                </h2>
                <div className="flex space-x-reverse space-x-3">
                  <button
                    onClick={() => exportToCSV('dns')}
                    className="flex items-center px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors"
                  >
                    <Download size={16} className="ml-2" />
                    export CSV
                  </button>
                  <button
                    onClick={() => openDnsModal()}
                    className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                  >
                    <Plus size={16} className="ml-2" />
                    افزودن رکورد DNS
                  </button>
                </div>
              </div>
              <div className="space-y-4">
                {dnsRecords.length > 0 ? (
                  dnsRecords.map((dns) => (
                    <div key={dns.id} className="border rounded-lg p-4 hover:shadow-md transition-shadow">
                      <div className="flex justify-between items-start">
                        <div className="flex-1">
                          <div className="flex items-center">
                            <h4 className="font-bold text-lg">{dns.domain}</h4>
                            <span className="mr-3 px-2 py-1 bg-blue-100 text-blue-800 rounded-full text-xs font-medium">
                              {dns.type}
                            </span>
                            {dns.proxy_status && (
                              <Shield size={16} className="mr-2 text-green-600" title="Proxy فعال" />
                            )}
                          </div>
                          <div className="flex items-center mt-2 text-sm text-gray-500">
                            <Globe2 size={16} className="ml-2" />
                            <span>{dns.value}</span>
                            <button
                              onClick={() => copyToClipboard(dns.value)}
                              className="mr-2 p-1 hover:bg-gray-100 rounded"
                              title="کپی مقدار"
                            >
                              <Copy size={12} />
                            </button>
                          </div>
                          <div className="flex items-center mt-2 text-sm text-gray-500">
                            <Cloud size={16} className="ml-2" />
                            <span>{dns.cloudflare_account}</span>
                            <span className="mx-2">•</span>
                            <Mail size={14} className="ml-1" />
                            <span>{dns.email}</span>
                          </div>
                          <div className="text-xs text-gray-500 mt-1">
                            TTL: {dns.ttl || 3600} • {formatDate(dns.created_at)}
                          </div>
                        </div>
                        <div className="flex items-center space-x-reverse space-x-2">
                          <button
                            onClick={() => openDnsModal(dns)}
                            className="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
                          >
                            <Edit size={18} />
                          </button>
                          <button
                            onClick={() => deleteDns(dns.id)}
                            className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors"
                          >
                            <Trash2 size={18} />
                          </button>
                        </div>
                      </div>
                    </div>
                  ))
                ) : (
                  <p className="text-gray-500 text-center py-8">هیچ رکورد DNS یافت نشد</p>
                )}
              </div>
            </div>
          )}

          {/* IP Management Page */}
          {currentPage === 'ip-management' && !loading && (
            <div className="bg-white rounded-xl shadow-lg p-6">
              <h2 className="text-2xl font-bold text-gray-800 mb-6 flex items-center">
                <Network className="ml-2" />
                مدیریت IP ها
              </h2>
              <div className="space-y-4">
                {ipDistribution.length > 0 ? (
                  ipDistribution.map((ip) => (
                    <div key={ip.ip_address} className="border rounded-lg p-6">
                      <div className="flex justify-between items-center mb-4">
                        <h3 className="text-xl font-bold flex items-center">
                          {ip.ip_address}
                          <button
                            onClick={() => copyToClipboard(ip.ip_address)}
                            className="mr-2 p-1 hover:bg-gray-100 rounded"
                            title="کپی آدرس IP"
                          >
                            <Copy size={16} />
                          </button>
                        </h3>
                        <div className="flex items-center space-x-reverse space-x-4">
                          <span className="text-sm text-gray-600">
                            <Globe size={16} className="inline ml-1" />
                            {ip.total_websites} سایت
                          </span>
                          <span className="text-sm text-gray-600">
                            <Server size={16} className="inline ml-1" />
                            {ip.total_hosts} هاست
                          </span>
                          <span className="text-lg font-bold text-blue-600">{ip.total_items} مجموع</span>
                        </div>
                      </div>
                      <div className="text-center text-gray-500">
                        برای مشاهده جزئیات کامل، به صفحات سایت‌ها و هاست‌ها مراجعه کنید
                      </div>
                    </div>
                  ))
                ) : (
                  <p className="text-gray-500 text-center py-8">هیچ IP یافت نشد</p>
                )}
              </div>
            </div>
          )}

          {/* Security Page */}
          {currentPage === 'security' && !loading && (
            <div className="bg-white rounded-xl shadow-lg p-6">
              <h2 className="text-2xl font-bold text-gray-800 mb-6 flex items-center">
                <Shield className="ml-2" />
                امنیت و تنظیمات
              </h2>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="border rounded-lg p-6">
                  <h3 className="text-lg font-semibold mb-4 flex items-center">
                    <Lock className="ml-2" />
                    امنیت داده‌ها
                  </h3>
                  <div className="space-y-3">
                    <div className="flex justify-between items-center">
                      <span>رمزنگاری داده‌ها</span>
                      <span className="text-green-600">فعال</span>
                    </div>
                    <div className="flex justify-between items-center">
                      <span>پشتیبان‌گیری خودکار</span>
                      <span className="text-green-600">فعال</span>
                    </div>
                    <div className="flex justify-between items-center">
                      <span>لاگ امنیتی</span>
                      <span className="text-green-600">فعال</span>
                    </div>
                  </div>
                </div>
                <div className="border rounded-lg p-6">
                  <h3 className="text-lg font-semibold mb-4 flex items-center">
                    <Settings className="ml-2" />
                    تنظیمات سیستم
                  </h3>
                  <div className="space-y-3">
                    <div className="flex justify-between items-center">
                      <span>نسخه سیستم</span>
                      <span>2.0.0</span>
                    </div>
                    <div className="flex justify-between items-center">
                      <span>آخرین به‌روزرسانی</span>
                      <span>{formatDate(new Date().toISOString())}</span>
                    </div>
                    <div className="flex justify-between items-center">
                      <span>وضعیت سرور</span>
                      <span className="text-green-600">سالم</span>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          )}
        </main>
      </div>

      {/* Enhanced Website Modal */}
      {showWebsiteModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-xl p-6 w-full max-w-md mx-4 max-h-[90vh] overflow-y-auto">
            <h3 className="text-xl font-bold mb-4">
              {editingWebsite ? 'ویرایش سایت' : 'افزودن سایت جدید'}
            </h3>
            <div className="space-y-4">
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">نام سایت</label>
                <input
                  type="text"
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={websiteForm.name}
                  onChange={(e) => setWebsiteForm({ ...websiteForm, name: e.target.value })}
                  required
                />
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">دامنه</label>
                <input
                  type="text"
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={websiteForm.domain}
                  onChange={(e) => setWebsiteForm({ ...websiteForm, domain: e.target.value })}
                  required
                />
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">آدرس IP</label>
                <input
                  type="text"
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={websiteForm.ip_address}
                  onChange={(e) => setWebsiteForm({ ...websiteForm, ip_address: e.target.value })}
                  required
                />
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">وضعیت</label>
                <select
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={websiteForm.status}
                  onChange={(e) => setWebsiteForm({ ...websiteForm, status: e.target.value as any })}
                >
                  <option value="active">فعال</option>
                  <option value="inactive">غیرفعال</option>
                  <option value="suspended">مسدود</option>
                  <option value="redirected">ریدایرکت شده</option>
                </select>
              </div>
              {websiteForm.status === 'redirected' && (
                <div>
                  <label className="block text-gray-700 text-sm font-bold mb-2">آدرس ریدایرکت</label>
                  <input
                    type="url"
                    className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                    value={websiteForm.redirect_url}
                    onChange={(e) => setWebsiteForm({ ...websiteForm, redirect_url: e.target.value })}
                    placeholder="https://example.com"
                  />
                </div>
              )}
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">توضیحات</label>
                <textarea
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  rows={3}
                  value={websiteForm.description}
                  onChange={(e) => setWebsiteForm({ ...websiteForm, description: e.target.value })}
                />
              </div>
            </div>
            <div className="flex justify-end space-x-reverse space-x-3 mt-6">
              <button
                onClick={() => {
                  setShowWebsiteModal(false);
                  resetWebsiteForm();
                }}
                className="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
              >
                انصراف
              </button>
              <button
                onClick={saveWebsite}
                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
              >
                ذخیره
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Enhanced Host Modal */}
      {showHostModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-xl p-6 w-full max-w-md mx-4 max-h-[90vh] overflow-y-auto">
            <h3 className="text-xl font-bold mb-4">
              {editingHost ? 'ویرایش هاست' : 'افزودن هاست جدید'}
            </h3>
            <div className="space-y-4">
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">نام هاست</label>
                <input
                  type="text"
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={hostForm.name}
                  onChange={(e) => setHostForm({ ...hostForm, name: e.target.value })}
                  required
                />
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">آدرس IP</label>
                <input
                  type="text"
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={hostForm.ip_address}
                  onChange={(e) => setHostForm({ ...hostForm, ip_address: e.target.value })}
                  required
                />
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">ارائه‌دهنده</label>
                <input
                  type="text"
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={hostForm.provider}
                  onChange={(e) => setHostForm({ ...hostForm, provider: e.target.value })}
                />
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">موقعیت</label>
                <input
                  type="text"
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={hostForm.location}
                  onChange={(e) => setHostForm({ ...hostForm, location: e.target.value })}
                />
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">مشخصات</label>
                <textarea
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  rows={3}
                  value={hostForm.specs}
                  onChange={(e) => setHostForm({ ...hostForm, specs: e.target.value })}
                />
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">وضعیت</label>
                <select
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={hostForm.status}
                  onChange={(e) => setHostForm({ ...hostForm, status: e.target.value as any })}
                >
                  <option value="active">فعال</option>
                  <option value="inactive">غیرفعال</option>
                  <option value="maintenance">در حال تعمیر</option>
                </select>
              </div>
            </div>
            <div className="flex justify-end space-x-reverse space-x-3 mt-6">
              <button
                onClick={() => {
                  setShowHostModal(false);
                  resetHostForm();
                }}
                className="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
              >
                انصراف
              </button>
              <button
                onClick={saveHost}
                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
              >
                ذخیره
              </button>
            </div>
          </div>
        </div>
      )}

      {/* DNS Modal */}
      {showDnsModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-xl p-6 w-full max-w-md mx-4 max-h-[90vh] overflow-y-auto">
            <h3 className="text-xl font-bold mb-4">
              {editingDns ? 'ویرایش رکورد DNS' : 'افزودن رکورد DNS جدید'}
            </h3>
            <div className="space-y-4">
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">دامنه</label>
                <input
                  type="text"
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={dnsForm.domain}
                  onChange={(e) => setDnsForm({ ...dnsForm, domain: e.target.value })}
                  required
                />
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">نوع رکورد</label>
                <select
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={dnsForm.type}
                  onChange={(e) => setDnsForm({ ...dnsForm, type: e.target.value as any })}
                >
                  <option value="A">A</option>
                  <option value="AAAA">AAAA</option>
                  <option value="CNAME">CNAME</option>
                  <option value="MX">MX</option>
                  <option value="TXT">TXT</option>
                  <option value="NS">NS</option>
                  <option value="SRV">SRV</option>
                </select>
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">مقدار</label>
                <input
                  type="text"
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={dnsForm.value}
                  onChange={(e) => setDnsForm({ ...dnsForm, value: e.target.value })}
                  required
                />
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">اکانت Cloudflare</label>
                <input
                  type="text"
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={dnsForm.cloudflare_account}
                  onChange={(e) => setDnsForm({ ...dnsForm, cloudflare_account: e.target.value })}
                  required
                />
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">ایمیل</label>
                <input
                  type="email"
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={dnsForm.email}
                  onChange={(e) => setDnsForm({ ...dnsForm, email: e.target.value })}
                  required
                />
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">رمز عبور</label>
                <div className="relative">
                  <input
                    type={showPassword[dnsForm.cloudflare_account] ? 'text' : 'password'}
                    className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                    value={dnsForm.password}
                    onChange={(e) => setDnsForm({ ...dnsForm, password: e.target.value })}
                  />
                  <button
                    type="button"
                    onClick={() => setShowPassword({ ...showPassword, [dnsForm.cloudflare_account]: !showPassword[dnsForm.cloudflare_account] })}
                    className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-500"
                  >
                    {showPassword[dnsForm.cloudflare_account] ? <EyeOff size={20} /> : <Eye size={20} />}
                  </button>
                </div>
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">TTL</label>
                <input
                  type="number"
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={dnsForm.ttl}
                  onChange={(e) => setDnsForm({ ...dnsForm, ttl: parseInt(e.target.value) })}
                  min={60}
                  max={86400}
                />
              </div>
              <div className="flex items-center">
                <input
                  type="checkbox"
                  id="proxy_status"
                  className="ml-2"
                  checked={dnsForm.proxy_status}
                  onChange={(e) => setDnsForm({ ...dnsForm, proxy_status: e.target.checked })}
                />
                <label htmlFor="proxy_status" className="text-gray-700 text-sm font-bold">
                  فعال بودن Proxy (Cloudflare)
                </label>
              </div>
            </div>
            <div className="flex justify-end space-x-reverse space-x-3 mt-6">
              <button
                onClick={() => {
                  setShowDnsModal(false);
                  resetDnsForm();
                }}
                className="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
              >
                انصراف
              </button>
              <button
                onClick={saveDns}
                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
              >
                ذخیره
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Batch Import Modal */}
      {showBatchImportModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-xl p-6 w-full max-w-2xl mx-4 max-h-[90vh] overflow-y-auto">
            <h3 className="text-xl font-bold mb-4">وارد کردن دسته جمعی</h3>
            <div className="mb-4">
              <label className="block text-gray-700 text-sm font-bold mb-2">نوع داده</label>
              <select
                className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                value={importType}
                onChange={(e) => setImportType(e.target.value as 'websites' | 'hosts')}
              >
                <option value="websites">سایت‌ها</option>
                <option value="hosts">هاست‌ها</option>
              </select>
            </div>
            <div className="mb-4">
              <label className="block text-gray-700 text-sm font-bold mb-2">
                {importType === 'websites' ? 'فرمت: نام,دامنه,آدرس IP,توضیحات' : 'فرمت: نام,آدرس IP,ارائه‌دهنده,موقعیت,مشخصات'}
              </label>
              <textarea
                className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                rows={10}
                value={batchImportData}
                onChange={(e) => setBatchImportData(e.target.value)}
                placeholder={importType === 'websites' 
                  ? "سایت اصلی,main-company.com,192.168.1.100,وبسایت اصلی\nفروشگاه,shop.example.com,192.168.1.101,فروشگاه آنلاین"
                  : "سرور اصلی,192.168.1.100,DigitalOcean,آمستردام,CPU: 4Core, RAM: 8GB\nسرور فروشگاه,192.168.1.101,Vultr,فرانکفورت,CPU: 2Core, RAM: 4GB"
                }
              />
            </div>
            <div className="mb-4">
              <input
                ref={fileInputRef}
                type="file"
                accept=".csv"
                onChange={handleFileUpload}
                className="hidden"
              />
              <button
                onClick={() => fileInputRef.current?.click()}
                className="flex items-center px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700 transition-colors"
              >
                <Upload size={16} className="ml-2" />
                انتخاب فایل CSV
              </button>
            </div>
            <div className="flex justify-end space-x-reverse space-x-3">
              <button
                onClick={() => {
                  setShowBatchImportModal(false);
                  setBatchImportData('');
                }}
                className="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
              >
                انصراف
              </button>
              <button
                onClick={handleBatchImport}
                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
              >
                وارد کردن
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Filter Modal */}
      {showFilterModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-xl p-6 w-full max-w-md mx-4 max-h-[90vh] overflow-y-auto">
            <h3 className="text-xl font-bold mb-4">فیلتر پیشرفته</h3>
            <div className="space-y-4">
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">وضعیت</label>
                <div className="space-y-2">
                  {['active', 'inactive', 'suspended', 'redirected', 'maintenance'].map((status) => (
                    <label key={status} className="flex items-center">
                      <input
                        type="checkbox"
                        className="ml-2"
                        checked={filters.status.includes(status)}
                        onChange={(e) => {
                          if (e.target.checked) {
                            setFilters({ ...filters, status: [...filters.status, status] });
                          } else {
                            setFilters({ ...filters, status: filters.status.filter(s => s !== status) });
                          }
                        }}
                      />
                      <span className="text-sm">
                        {status === 'active' ? 'فعال' : 
                         status === 'inactive' ? 'غیرفعال' : 
                         status === 'suspended' ? 'مسدود' : 
                         status === 'redirected' ? 'ریدایرکت شده' : 'در حال تعمیر'}
                      </span>
                    </label>
                  ))}
                </div>
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">آدرس IP</label>
                <input
                  type="text"
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={filters.ip_address}
                  onChange={(e) => setFilters({ ...filters, ip_address: e.target.value })}
                  placeholder="192.168.1.1"
                />
              </div>
              <div>
                <label className="block text-gray-700 text-sm font-bold mb-2">ارائه‌دهنده</label>
                <input
                  type="text"
                  className="w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  value={filters.provider}
                  onChange={(e) => setFilters({ ...filters, provider: e.target.value })}
                  placeholder="DigitalOcean"
                />
              </div>
            </div>
            <div className="flex justify-between space-x-reverse space-x-3 mt-6">
              <button
                onClick={clearFilters}
                className="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
              >
                پاک کردن فیلترها
              </button>
              <div className="space-x-reverse space-x-3">
                <button
                  onClick={() => setShowFilterModal(false)}
                  className="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
                >
                  انصراف
                </button>
                <button
                  onClick={applyFilters}
                  className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                >
                  اعمال فیلترها
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}