import { NextRequest, NextResponse } from 'next/server';

// Mock data for demonstration
let mockWebsites = [
  {
    id: 1,
    name: 'سایت اصلی شرکت',
    domain: 'main-company.com',
    ip_address: '192.168.1.100',
    description: 'وبسایت اصلی شرکت با تمام خدمات',
    status: 'active',
    redirect_url: '',
    created_at: new Date().toISOString(),
    tags: ['شرکتی', 'اصلی']
  },
  {
    id: 2,
    name: 'فروشگاه آنلاین',
    domain: 'shop.example.com',
    ip_address: '192.168.1.101',
    description: 'فروشگاه آنلاین محصولات',
    status: 'active',
    redirect_url: '',
    created_at: new Date().toISOString(),
    tags: ['فروشگاه', 'تجارت']
  },
  {
    id: 3,
    name: 'وبلاگ فنی',
    domain: 'blog.techsite.com',
    ip_address: '192.168.1.102',
    description: 'وبلاگ تخصصی فنی و آموزشی',
    status: 'redirected',
    redirect_url: 'https://newblog.techsite.com',
    created_at: new Date().toISOString(),
    tags: ['وبلاگ', 'فنی', 'آموزشی']
  },
  {
    id: 4,
    name: 'پورتال مشتریان',
    domain: 'portal.client.com',
    ip_address: '192.168.1.103',
    description: 'پورتال مدیریت مشتریان',
    status: 'inactive',
    redirect_url: '',
    created_at: new Date().toISOString(),
    tags: ['مشتری', 'پورتال']
  }
];

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const id = searchParams.get('id');
    const ip = searchParams.get('ip');

    let data = mockWebsites;

    if (id) {
      data = mockWebsites.find(website => website.id === parseInt(id)) || null;
    } else if (ip) {
      data = mockWebsites.filter(website => website.ip_address === ip);
    }

    return NextResponse.json({
      success: true,
      message: 'Websites retrieved successfully',
      data
    });
  } catch (error) {
    console.error('Websites API Error:', error);
    return NextResponse.json(
      {
        success: false,
        message: 'Internal server error',
        error: error instanceof Error ? error.message : 'Unknown error'
      },
      { status: 500 }
    );
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    
    // Validate required fields
    if (!body.name || !body.domain || !body.ip_address) {
      return NextResponse.json(
        {
          success: false,
          message: 'Missing required fields: name, domain, ip_address'
        },
        { status: 400 }
      );
    }

    const newWebsite = {
      id: Math.max(...mockWebsites.map(w => w.id)) + 1,
      name: body.name,
      domain: body.domain,
      ip_address: body.ip_address,
      description: body.description || '',
      status: body.status || 'active',
      redirect_url: body.redirect_url || '',
      tags: body.tags || [],
      created_at: new Date().toISOString()
    };

    mockWebsites.push(newWebsite);

    return NextResponse.json({
      success: true,
      message: 'Website created successfully',
      data: newWebsite
    });
  } catch (error) {
    console.error('Websites POST Error:', error);
    return NextResponse.json(
      {
        success: false,
        message: 'Internal server error',
        error: error instanceof Error ? error.message : 'Unknown error'
      },
      { status: 500 }
    );
  }
}

export async function PUT(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const id = searchParams.get('id');
    
    if (!id) {
      return NextResponse.json(
        {
          success: false,
          message: 'Website ID is required'
        },
        { status: 400 }
      );
    }

    const body = await request.json();
    const websiteIndex = mockWebsites.findIndex(website => website.id === parseInt(id));
    
    if (websiteIndex === -1) {
      return NextResponse.json(
        {
          success: false,
          message: 'Website not found'
        },
        { status: 404 }
      );
    }

    mockWebsites[websiteIndex] = {
      ...mockWebsites[websiteIndex],
      ...body,
      updated_at: new Date().toISOString()
    };

    return NextResponse.json({
      success: true,
      message: 'Website updated successfully',
      data: mockWebsites[websiteIndex]
    });
  } catch (error) {
    console.error('Websites PUT Error:', error);
    return NextResponse.json(
      {
        success: false,
        message: 'Internal server error',
        error: error instanceof Error ? error.message : 'Unknown error'
      },
      { status: 500 }
    );
  }
}

export async function DELETE(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const id = searchParams.get('id');
    
    if (!id) {
      return NextResponse.json(
        {
          success: false,
          message: 'Website ID is required'
        },
        { status: 400 }
      );
    }

    const websiteIndex = mockWebsites.findIndex(website => website.id === parseInt(id));
    
    if (websiteIndex === -1) {
      return NextResponse.json(
        {
          success: false,
          message: 'Website not found'
        },
        { status: 404 }
      );
    }

    const deletedWebsite = mockWebsites.splice(websiteIndex, 1)[0];

    return NextResponse.json({
      success: true,
      message: 'Website deleted successfully',
      data: deletedWebsite
    });
  } catch (error) {
    console.error('Websites DELETE Error:', error);
    return NextResponse.json(
      {
        success: false,
        message: 'Internal server error',
        error: error instanceof Error ? error.message : 'Unknown error'
      },
      { status: 500 }
    );
  }
}