import { NextRequest, NextResponse } from 'next/server';

// System monitoring and health check API
export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const type = searchParams.get('type') || 'health';

    switch (type) {
      case 'health':
        return await healthCheck();
      case 'stats':
        return await getSystemStats();
      case 'performance':
        return await getPerformanceMetrics();
      case 'security':
        return await getSecurityStatus();
      default:
        return NextResponse.json(
          { success: false, message: 'Invalid monitoring type' },
          { status: 400 }
        );
    }
  } catch (error) {
    console.error('Monitoring API Error:', error);
    return NextResponse.json(
      { success: false, message: 'Monitoring failed', error: error instanceof Error ? error.message : 'Unknown error' },
      { status: 500 }
    );
  }
}

async function healthCheck() {
  const health = {
    status: 'healthy',
    timestamp: new Date().toISOString(),
    uptime: process.uptime(),
    version: process.env.npm_package_version || '1.0.0',
    environment: process.env.NODE_ENV || 'development',
    services: {
      api: await checkApiHealth(),
      database: await checkDatabaseHealth(),
      dns: await checkDnsHealth(),
      cloudflare: await checkCloudflareHealth()
    },
    resources: {
      memory: process.memoryUsage(),
      cpu: process.cpuUsage()
    }
  };

  // Determine overall health status
  const serviceStatuses = Object.values(health.services);
  const hasFailure = serviceStatuses.some((service: any) => service.status !== 'healthy');
  
  health.status = hasFailure ? 'degraded' : 'healthy';

  return NextResponse.json({
    success: true,
    data: health
  });
}

async function getSystemStats() {
  try {
    // Get stats from all APIs
    const [websitesResponse, hostsResponse, dnsResponse, dashboardResponse] = await Promise.all([
      fetch('http://localhost:3000/api/websites'),
      fetch('http://localhost:3000/api/hosts'),
      fetch('http://localhost:3000/api/dns'),
      fetch('http://localhost:3000/api/dashboard')
    ]);

    const [websitesData, hostsData, dnsData, dashboardData] = await Promise.all([
      websitesResponse.json(),
      hostsResponse.json(),
      dnsResponse.json(),
      dashboardResponse.json()
    ]);

    const stats = {
      overview: dashboardData.data?.stats || {},
      details: {
        websites: {
          total: websitesData.data?.length || 0,
          active: websitesData.data?.filter((w: any) => w.status === 'active').length || 0,
          inactive: websitesData.data?.filter((w: any) => w.status === 'inactive').length || 0,
          suspended: websitesData.data?.filter((w: any) => w.status === 'suspended').length || 0,
          redirected: websitesData.data?.filter((w: any) => w.status === 'redirected').length || 0,
        },
        hosts: {
          total: hostsData.data?.length || 0,
          active: hostsData.data?.filter((h: any) => h.status === 'active').length || 0,
          inactive: hostsData.data?.filter((h: any) => h.status === 'inactive').length || 0,
          maintenance: hostsData.data?.filter((h: any) => h.status === 'maintenance').length || 0,
        },
        dns: {
          total: dnsData.data?.length || 0,
          proxied: dnsData.data?.filter((d: any) => d.proxy_status).length || 0,
          types: dnsData.data?.reduce((acc: any, record: any) => {
            acc[record.type] = (acc[record.type] || 0) + 1;
            return acc;
          }, {}) || {}
        }
      },
      trends: {
        dailyGrowth: await calculateDailyGrowth(),
        topProviders: await getTopProviders(),
        ipDistribution: await getIpDistribution()
      }
    };

    return NextResponse.json({
      success: true,
      data: stats
    });
  } catch (error) {
    return NextResponse.json({
      success: false,
      message: 'Failed to get system stats',
      error: error instanceof Error ? error.message : 'Unknown error'
    }, { status: 500 });
  }
}

async function getPerformanceMetrics() {
  const metrics = {
    responseTime: await measureResponseTime(),
    throughput: await calculateThroughput(),
    errorRate: await calculateErrorRate(),
    resources: {
      memory: process.memoryUsage(),
      cpu: process.cpuUsage(),
      uptime: process.uptime()
    },
    endpoints: {
      '/api/websites': await getEndpointMetrics('/api/websites'),
      '/api/hosts': await getEndpointMetrics('/api/hosts'),
      '/api/dns': await getEndpointMetrics('/api/dns'),
      '/api/dashboard': await getEndpointMetrics('/api/dashboard')
    }
  };

  return NextResponse.json({
    success: true,
    data: metrics
  });
}

async function getSecurityStatus() {
  const security = {
    lastScan: new Date().toISOString(),
    vulnerabilities: await scanForVulnerabilities(),
    configurations: {
      https: true,
      cors: true,
      rateLimit: true,
      inputValidation: true,
      outputSanitization: true
    },
    recentActivity: await getRecentSecurityEvents(),
    recommendations: await getSecurityRecommendations()
  };

  return NextResponse.json({
    success: true,
    data: security
  });
}

// Helper functions
async function checkApiHealth() {
  try {
    const start = Date.now();
    const response = await fetch('http://localhost:3000/api/dashboard', { 
      method: 'GET',
      signal: AbortSignal.timeout(5000)
    });
    const responseTime = Date.now() - start;
    
    return {
      status: response.ok ? 'healthy' : 'unhealthy',
      responseTime,
      lastCheck: new Date().toISOString()
    };
  } catch (error) {
    return {
      status: 'unhealthy',
      error: error instanceof Error ? error.message : 'Unknown error',
      lastCheck: new Date().toISOString()
    };
  }
}

async function checkDatabaseHealth() {
  // Mock database health check
  return {
    status: 'healthy',
    connections: 5,
    maxConnections: 100,
    lastCheck: new Date().toISOString()
  };
}

async function checkDnsHealth() {
  try {
    const response = await fetch('http://localhost:3000/api/dns', { 
      method: 'GET',
      signal: AbortSignal.timeout(5000)
    });
    
    return {
      status: response.ok ? 'healthy' : 'unhealthy',
      recordsCount: response.ok ? (await response.json()).data?.length || 0 : 0,
      lastCheck: new Date().toISOString()
    };
  } catch (error) {
    return {
      status: 'unhealthy',
      error: error instanceof Error ? error.message : 'Unknown error',
      lastCheck: new Date().toISOString()
    };
  }
}

async function checkCloudflareHealth() {
  // Mock Cloudflare health check
  return {
    status: 'healthy',
    accountsConnected: 4,
    lastSync: new Date().toISOString(),
    lastCheck: new Date().toISOString()
  };
}

async function calculateDailyGrowth() {
  // Mock daily growth calculation
  return {
    websites: Math.floor(Math.random() * 5),
    hosts: Math.floor(Math.random() * 3),
    dns: Math.floor(Math.random() * 8)
  };
}

async function getTopProviders() {
  try {
    const response = await fetch('http://localhost:3000/api/hosts');
    const data = await response.json();
    
    const providers = data.data?.reduce((acc: any, host: any) => {
      if (host.provider) {
        acc[host.provider] = (acc[host.provider] || 0) + 1;
      }
      return acc;
    }, {}) || {};

    return Object.entries(providers)
      .sort(([,a], [,b]) => (b as number) - (a as number))
      .slice(0, 5);
  } catch (error) {
    return [];
  }
}

async function getIpDistribution() {
  try {
    const response = await fetch('http://localhost:3000/api/dashboard?type=ipDistribution');
    const data = await response.json();
    return data.data || [];
  } catch (error) {
    return [];
  }
}

async function measureResponseTime() {
  const start = Date.now();
  try {
    await fetch('http://localhost:3000/api/dashboard');
    return Date.now() - start;
  } catch (error) {
    return -1;
  }
}

async function calculateThroughput() {
  // Mock throughput calculation
  return Math.floor(Math.random() * 1000) + 500;
}

async function calculateErrorRate() {
  // Mock error rate calculation
  return Math.random() * 5; // percentage
}

async function getEndpointMetrics(endpoint: string) {
  const start = Date.now();
  try {
    const response = await fetch(`http://localhost:3000${endpoint}`);
    return {
      responseTime: Date.now() - start,
      status: response.status,
      healthy: response.ok
    };
  } catch (error) {
    return {
      responseTime: -1,
      status: 500,
      healthy: false,
      error: error instanceof Error ? error.message : 'Unknown error'
    };
  }
}

async function scanForVulnerabilities() {
  // Mock vulnerability scan
  return [
    {
      severity: 'low',
      description: 'Missing security headers on some endpoints',
      recommendation: 'Add comprehensive security headers'
    },
    {
      severity: 'info',
      description: 'Consider implementing rate limiting',
      recommendation: 'Add rate limiting middleware'
    }
  ];
}

async function getRecentSecurityEvents() {
  // Mock recent security events
  return [
    {
      timestamp: new Date().toISOString(),
      type: 'login_attempt',
      severity: 'info',
      description: 'Successful API key authentication'
    },
    {
      timestamp: new Date(Date.now() - 3600000).toISOString(),
      type: 'validation_error',
      severity: 'low',
      description: 'Invalid input detected and blocked'
    }
  ];
}

async function getSecurityRecommendations() {
  return [
    'Enable HTTPS for all API endpoints',
    'Implement rate limiting for API calls',
    'Add comprehensive logging for security events',
    'Regularly update dependencies',
    'Implement input validation on all endpoints',
    'Add CORS headers for better security'
  ];
}