import { NextRequest, NextResponse } from 'next/server';
import { validateInput, sanitizeInput } from '@/lib/security';

// CSV export functionality
export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const type = searchParams.get('type');
    
    if (!type || !['websites', 'hosts', 'dns'].includes(type)) {
      return NextResponse.json(
        { success: false, message: 'Invalid export type' },
        { status: 400 }
      );
    }

    let data: any[] = [];
    let filename = '';
    let headers: string[] = [];

    // Fetch data based on type
    switch (type) {
      case 'websites':
        const websitesResponse = await fetch(`${process.env.NEXTAUTH_URL || 'http://localhost:3000'}/api/websites`);
        const websitesData = await websitesResponse.json();
        data = websitesData.data || [];
        filename = `websites-export-${new Date().toISOString().split('T')[0]}.csv`;
        headers = ['ID', 'Name', 'Domain', 'IP Address', 'Status', 'Redirect URL', 'Description', 'Tags', 'Created At'];
        break;
        
      case 'hosts':
        const hostsResponse = await fetch(`${process.env.NEXTAUTH_URL || 'http://localhost:3000'}/api/hosts`);
        const hostsData = await hostsResponse.json();
        data = hostsData.data || [];
        filename = `hosts-export-${new Date().toISOString().split('T')[0]}.csv`;
        headers = ['ID', 'Name', 'IP Address', 'Provider', 'Location', 'Status', 'Specs', 'Tags', 'Created At'];
        break;
        
      case 'dns':
        const dnsResponse = await fetch(`${process.env.NEXTAUTH_URL || 'http://localhost:3000'}/api/dns`);
        const dnsData = await dnsResponse.json();
        data = dnsData.data || [];
        filename = `dns-export-${new Date().toISOString().split('T')[0]}.csv`;
        headers = ['ID', 'Domain', 'Type', 'Value', 'Cloudflare Account', 'Email', 'TTL', 'Proxy Status', 'Created At'];
        break;
    }

    // Generate CSV content
    const csvContent = [
      headers.join(','),
      ...data.map(item => {
        switch (type) {
          case 'websites':
            return `${item.id},"${item.name}","${item.domain}","${item.ip_address}","${item.status}","${item.redirect_url || ''}","${item.description || ''}","${(item.tags || []).join(';')}","${item.created_at}"`;
          case 'hosts':
            return `${item.id},"${item.name}","${item.ip_address}","${item.provider || ''}","${item.location || ''}","${item.status}","${item.specs || ''}","${(item.tags || []).join(';')}","${item.created_at}"`;
          case 'dns':
            return `${item.id},"${item.domain}","${item.type}","${item.value}","${item.cloudflare_account}","${item.email}",${item.ttl || 3600},"${item.proxy_status}","${item.created_at}"`;
          default:
            return '';
        }
      })
    ].join('\n');

    // Create CSV file
    const blob = new Blob(['\ufeff' + csvContent], { type: 'text/csv;charset=utf-8;' });
    const csvBuffer = await blob.arrayBuffer();
    
    return new NextResponse(csvBuffer, {
      status: 200,
      headers: {
        'Content-Type': 'text/csv; charset=utf-8',
        'Content-Disposition': `attachment; filename="${filename}"`,
        'Content-Length': csvBuffer.byteLength.toString(),
      },
    });
    
  } catch (error) {
    console.error('Export Error:', error);
    return NextResponse.json(
      { success: false, message: 'Export failed', error: error instanceof Error ? error.message : 'Unknown error' },
      { status: 500 }
    );
  }
}

// CSV import functionality
export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { type, data } = body;

    if (!type || !['websites', 'hosts'].includes(type)) {
      return NextResponse.json(
        { success: false, message: 'Invalid import type' },
        { status: 400 }
      );
    }

    if (!data || typeof data !== 'string') {
      return NextResponse.json(
        { success: false, message: 'Invalid CSV data' },
        { status: 400 }
      );
    }

    // Parse CSV
    const lines = data.trim().split('\n');
    const headers = lines[0].split(',').map(h => h.trim().toLowerCase());
    const results: any[] = [];
    const errors: string[] = [];

    // Validation rules
    const validationRules: any = {
      websites: {
        name: { required: true, type: 'string', minLength: 1, maxLength: 255 },
        domain: { required: true, type: 'string', domain: true },
        ip_address: { required: true, type: 'string', ip: true },
        description: { required: false, type: 'string', maxLength: 1000 },
        status: { required: false, type: 'string', enum: ['active', 'inactive', 'suspended', 'redirected'] },
        redirect_url: { required: false, type: 'string', url: true },
        tags: { required: false, type: 'string' }
      },
      hosts: {
        name: { required: true, type: 'string', minLength: 1, maxLength: 255 },
        ip_address: { required: true, type: 'string', ip: true },
        provider: { required: false, type: 'string', maxLength: 255 },
        location: { required: false, type: 'string', maxLength: 255 },
        specs: { required: false, type: 'string', maxLength: 1000 },
        status: { required: false, type: 'string', enum: ['active', 'inactive', 'maintenance'] },
        tags: { required: false, type: 'string' }
      }
    };

    // Process each line
    for (let i = 1; i < lines.length; i++) {
      try {
        const values = lines[i].split(',').map(v => v.trim().replace(/^"|"$/g, ''));
        const item: any = {};

        // Map values to object
        headers.forEach((header, index) => {
          if (values[index]) {
            item[header] = values[index];
          }
        });

        // Sanitize input
        const sanitizedItem = sanitizeInput(item);

        // Validate input
        const validationErrors = validateInput(sanitizedItem, validationRules[type]);
        
        if (validationErrors.length > 0) {
          errors.push(`Line ${i + 1}: ${validationErrors.join(', ')}`);
          continue;
        }

        // Process tags
        if (sanitizedItem.tags) {
          sanitizedItem.tags = sanitizedItem.tags.split(';').map((tag: string) => tag.trim()).filter((tag: string) => tag);
        } else {
          sanitizedItem.tags = [];
        }

        // Set default status if not provided
        if (!sanitizedItem.status) {
          sanitizedItem.status = 'active';
        }

        results.push(sanitizedItem);
      } catch (error) {
        errors.push(`Line ${i + 1}: Invalid format - ${error instanceof Error ? error.message : 'Unknown error'}`);
      }
    }

    if (errors.length > 0) {
      return NextResponse.json({
        success: false,
        message: 'Validation errors found',
        errors,
        validCount: results.length,
        totalCount: lines.length - 1
      }, { status: 400 });
    }

    // Import data to respective APIs
    let importResults: any[] = [];
    const apiUrl = type === 'websites' ? '/api/websites' : '/api/hosts';

    for (const item of results) {
      try {
        const response = await fetch(`${process.env.NEXTAUTH_URL || 'http://localhost:3000'}${apiUrl}`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(item)
        });

        const result = await response.json();
        if (result.success) {
          importResults.push(result.data);
        } else {
          errors.push(`Failed to import ${item.name || item.domain}: ${result.message}`);
        }
      } catch (error) {
        errors.push(`Failed to import ${item.name || item.domain}: ${error instanceof Error ? error.message : 'Unknown error'}`);
      }
    }

    return NextResponse.json({
      success: true,
      message: `Successfully imported ${importResults.length} ${type}`,
      data: importResults,
      errors: errors.length > 0 ? errors : undefined,
      summary: {
        total: lines.length - 1,
        valid: results.length,
        imported: importResults.length,
        failed: errors.length
      }
    });

  } catch (error) {
    console.error('Import Error:', error);
    return NextResponse.json(
      { success: false, message: 'Import failed', error: error instanceof Error ? error.message : 'Unknown error' },
      { status: 500 }
    );
  }
}