import { NextRequest, NextResponse } from 'next/server';

// Mock data for demonstration
let mockHosts = [
  {
    id: 1,
    name: 'سرور اصلی',
    ip_address: '192.168.1.100',
    provider: 'DigitalOcean',
    location: 'آمستردام',
    specs: 'CPU: 4Core, RAM: 8GB, SSD: 160GB',
    status: 'active',
    created_at: new Date().toISOString(),
    tags: ['اصلی', 'پروداکشن']
  },
  {
    id: 2,
    name: 'سرور فروشگاه',
    ip_address: '192.168.1.101',
    provider: 'Vultr',
    location: 'فرانکفورت',
    specs: 'CPU: 2Core, RAM: 4GB, SSD: 80GB',
    status: 'active',
    created_at: new Date().toISOString(),
    tags: ['فروشگاه', 'تجارت']
  },
  {
    id: 3,
    name: 'سرور پشتیبان',
    ip_address: '192.168.1.200',
    provider: 'Hetzner',
    location: 'نورنبرگ',
    specs: 'CPU: 8Core, RAM: 16GB, SSD: 320GB',
    status: 'maintenance',
    created_at: new Date().toISOString(),
    tags: ['پشتیبان', 'بکاپ']
  },
  {
    id: 4,
    name: 'سرور توسعه',
    ip_address: '192.168.1.201',
    provider: 'Linode',
    location: 'لندن',
    specs: 'CPU: 2Core, RAM: 2GB, SSD: 40GB',
    status: 'active',
    created_at: new Date().toISOString(),
    tags: ['توسعه', 'تستی']
  }
];

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const id = searchParams.get('id');
    const ip = searchParams.get('ip');

    let data = mockHosts;

    if (id) {
      data = mockHosts.find(host => host.id === parseInt(id)) || null;
    } else if (ip) {
      data = mockHosts.filter(host => host.ip_address === ip);
    }

    return NextResponse.json({
      success: true,
      message: 'Hosts retrieved successfully',
      data
    });
  } catch (error) {
    console.error('Hosts API Error:', error);
    return NextResponse.json(
      {
        success: false,
        message: 'Internal server error',
        error: error instanceof Error ? error.message : 'Unknown error'
      },
      { status: 500 }
    );
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    
    // Validate required fields
    if (!body.name || !body.ip_address) {
      return NextResponse.json(
        {
          success: false,
          message: 'Missing required fields: name, ip_address'
        },
        { status: 400 }
      );
    }

    const newHost = {
      id: Math.max(...mockHosts.map(h => h.id)) + 1,
      name: body.name,
      ip_address: body.ip_address,
      provider: body.provider || '',
      location: body.location || '',
      specs: body.specs || '',
      status: body.status || 'active',
      tags: body.tags || [],
      created_at: new Date().toISOString()
    };

    mockHosts.push(newHost);

    return NextResponse.json({
      success: true,
      message: 'Host created successfully',
      data: newHost
    });
  } catch (error) {
    console.error('Hosts POST Error:', error);
    return NextResponse.json(
      {
        success: false,
        message: 'Internal server error',
        error: error instanceof Error ? error.message : 'Unknown error'
      },
      { status: 500 }
    );
  }
}

export async function PUT(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const id = searchParams.get('id');
    
    if (!id) {
      return NextResponse.json(
        {
          success: false,
          message: 'Host ID is required'
        },
        { status: 400 }
      );
    }

    const body = await request.json();
    const hostIndex = mockHosts.findIndex(host => host.id === parseInt(id));
    
    if (hostIndex === -1) {
      return NextResponse.json(
        {
          success: false,
          message: 'Host not found'
        },
        { status: 404 }
      );
    }

    mockHosts[hostIndex] = {
      ...mockHosts[hostIndex],
      ...body,
      updated_at: new Date().toISOString()
    };

    return NextResponse.json({
      success: true,
      message: 'Host updated successfully',
      data: mockHosts[hostIndex]
    });
  } catch (error) {
    console.error('Hosts PUT Error:', error);
    return NextResponse.json(
      {
        success: false,
        message: 'Internal server error',
        error: error instanceof Error ? error.message : 'Unknown error'
      },
      { status: 500 }
    );
  }
}

export async function DELETE(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const id = searchParams.get('id');
    
    if (!id) {
      return NextResponse.json(
        {
          success: false,
          message: 'Host ID is required'
        },
        { status: 400 }
      );
    }

    const hostIndex = mockHosts.findIndex(host => host.id === parseInt(id));
    
    if (hostIndex === -1) {
      return NextResponse.json(
        {
          success: false,
          message: 'Host not found'
        },
        { status: 404 }
      );
    }

    const deletedHost = mockHosts.splice(hostIndex, 1)[0];

    return NextResponse.json({
      success: true,
      message: 'Host deleted successfully',
      data: deletedHost
    });
  } catch (error) {
    console.error('Hosts DELETE Error:', error);
    return NextResponse.json(
      {
        success: false,
        message: 'Internal server error',
        error: error instanceof Error ? error.message : 'Unknown error'
      },
      { status: 500 }
    );
  }
}