import { NextRequest, NextResponse } from 'next/server';

// Mock DNS records data
let mockDnsRecords = [
  {
    id: 1,
    domain: 'main-company.com',
    type: 'A',
    value: '192.168.1.100',
    cloudflare_account: 'company-main',
    email: 'admin@company.com',
    password: 'encrypted_password_123',
    ttl: 3600,
    proxy_status: true,
    created_at: new Date().toISOString()
  },
  {
    id: 2,
    domain: 'shop.example.com',
    type: 'A',
    value: '192.168.1.101',
    cloudflare_account: 'shop-account',
    email: 'dns@shop.com',
    password: 'encrypted_password_456',
    ttl: 7200,
    proxy_status: false,
    created_at: new Date().toISOString()
  },
  {
    id: 3,
    domain: 'blog.techsite.com',
    type: 'CNAME',
    value: 'techsite.netlify.app',
    cloudflare_account: 'blog-account',
    email: 'tech@blog.com',
    password: 'encrypted_password_789',
    ttl: 1800,
    proxy_status: true,
    created_at: new Date().toISOString()
  },
  {
    id: 4,
    domain: 'portal.client.com',
    type: 'A',
    value: '192.168.1.103',
    cloudflare_account: 'client-portal',
    email: 'support@client.com',
    password: 'encrypted_password_012',
    ttl: 5400,
    proxy_status: false,
    created_at: new Date().toISOString()
  }
];

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const id = searchParams.get('id');
    const domain = searchParams.get('domain');
    const cloudflare_account = searchParams.get('cloudflare_account');

    let data = mockDnsRecords;

    if (id) {
      data = mockDnsRecords.find(record => record.id === parseInt(id)) || null;
    } else if (domain) {
      data = mockDnsRecords.filter(record => record.domain.includes(domain));
    } else if (cloudflare_account) {
      data = mockDnsRecords.filter(record => record.cloudflare_account.includes(cloudflare_account));
    }

    // Remove passwords from response for security
    if (Array.isArray(data)) {
      data = data.map(record => ({
        ...record,
        password: record.password ? '••••••••' : undefined
      }));
    } else if (data && data.password) {
      data = {
        ...data,
        password: '••••••••'
      };
    }

    return NextResponse.json({
      success: true,
      message: 'DNS records retrieved successfully',
      data
    });
  } catch (error) {
    console.error('DNS API Error:', error);
    return NextResponse.json(
      {
        success: false,
        message: 'Internal server error',
        error: error instanceof Error ? error.message : 'Unknown error'
      },
      { status: 500 }
    );
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    
    // Validate required fields
    if (!body.domain || !body.type || !body.value || !body.cloudflare_account || !body.email) {
      return NextResponse.json(
        {
          success: false,
          message: 'Missing required fields: domain, type, value, cloudflare_account, email'
        },
        { status: 400 }
      );
    }

    // Validate DNS record type
    const validTypes = ['A', 'AAAA', 'CNAME', 'MX', 'TXT', 'NS', 'SRV'];
    if (!validTypes.includes(body.type)) {
      return NextResponse.json(
        {
          success: false,
          message: 'Invalid DNS record type'
        },
        { status: 400 }
      );
    }

    // Encrypt password (in real app, use proper encryption)
    const encryptedPassword = body.password ? `encrypted_${Date.now()}_${body.password.length}` : '';

    const newDnsRecord = {
      id: Math.max(...mockDnsRecords.map(r => r.id)) + 1,
      domain: body.domain,
      type: body.type,
      value: body.value,
      cloudflare_account: body.cloudflare_account,
      email: body.email,
      password: encryptedPassword,
      ttl: body.ttl || 3600,
      proxy_status: body.proxy_status || false,
      created_at: new Date().toISOString()
    };

    mockDnsRecords.push(newDnsRecord);

    // Remove password from response
    const responseRecord = {
      ...newDnsRecord,
      password: '••••••••'
    };

    return NextResponse.json({
      success: true,
      message: 'DNS record created successfully',
      data: responseRecord
    });
  } catch (error) {
    console.error('DNS POST Error:', error);
    return NextResponse.json(
      {
        success: false,
        message: 'Internal server error',
        error: error instanceof Error ? error.message : 'Unknown error'
      },
      { status: 500 }
    );
  }
}

export async function PUT(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const id = searchParams.get('id');
    
    if (!id) {
      return NextResponse.json(
        {
          success: false,
          message: 'DNS record ID is required'
        },
        { status: 400 }
      );
    }

    const body = await request.json();
    const recordIndex = mockDnsRecords.findIndex(record => record.id === parseInt(id));
    
    if (recordIndex === -1) {
      return NextResponse.json(
        {
          success: false,
          message: 'DNS record not found'
        },
        { status: 404 }
      );
    }

    // Update record
    mockDnsRecords[recordIndex] = {
      ...mockDnsRecords[recordIndex],
      ...body,
      updated_at: new Date().toISOString()
    };

    // Remove password from response
    const responseRecord = {
      ...mockDnsRecords[recordIndex],
      password: mockDnsRecords[recordIndex].password ? '••••••••' : undefined
    };

    return NextResponse.json({
      success: true,
      message: 'DNS record updated successfully',
      data: responseRecord
    });
  } catch (error) {
    console.error('DNS PUT Error:', error);
    return NextResponse.json(
      {
        success: false,
        message: 'Internal server error',
        error: error instanceof Error ? error.message : 'Unknown error'
      },
      { status: 500 }
    );
  }
}

export async function DELETE(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const id = searchParams.get('id');
    
    if (!id) {
      return NextResponse.json(
        {
          success: false,
          message: 'DNS record ID is required'
        },
        { status: 400 }
      );
    }

    const recordIndex = mockDnsRecords.findIndex(record => record.id === parseInt(id));
    
    if (recordIndex === -1) {
      return NextResponse.json(
        {
          success: false,
          message: 'DNS record not found'
        },
        { status: 404 }
      );
    }

    const deletedRecord = mockDnsRecords.splice(recordIndex, 1)[0];

    return NextResponse.json({
      success: true,
      message: 'DNS record deleted successfully',
      data: {
        id: deletedRecord.id,
        domain: deletedRecord.domain,
        type: deletedRecord.type
      }
    });
  } catch (error) {
    console.error('DNS DELETE Error:', error);
    return NextResponse.json(
      {
        success: false,
        message: 'Internal server error',
        error: error instanceof Error ? error.message : 'Unknown error'
      },
      { status: 500 }
    );
  }
}