// API Base URL
const API_BASE = './api';

// App State
let currentPage = 'dashboard';
let isLoading = false;

// DOM Elements
const sidebarItems = document.querySelectorAll('.sidebar-item');
const pageContents = document.querySelectorAll('.page-content');
const globalSearch = document.getElementById('globalSearch');
const mobileMenuToggle = document.getElementById('mobileMenuToggle');
const sidebar = document.getElementById('sidebar');

// Initialize App
document.addEventListener('DOMContentLoaded', function() {
    initializeEventListeners();
    loadDashboard();
});

// Event Listeners
function initializeEventListeners() {
    // Sidebar navigation
    sidebarItems.forEach(item => {
        item.addEventListener('click', function() {
            const page = this.dataset.page;
            navigateToPage(page);
        });
    });

    // Mobile menu toggle
    mobileMenuToggle.addEventListener('click', function() {
        sidebar.classList.toggle('show');
    });

    // Global search
    globalSearch.addEventListener('input', debounce(function(e) {
        const query = e.target.value.trim();
        if (query.length > 2) {
            performSearch(query);
        } else if (query.length === 0) {
            loadCurrentPage();
        }
    }, 500));

    // Website form
    document.getElementById('websiteForm').addEventListener('submit', function(e) {
        e.preventDefault();
        saveWebsite();
    });

    // Host form
    document.getElementById('hostForm').addEventListener('submit', function(e) {
        e.preventDefault();
        saveHost();
    });

    // Add buttons
    document.getElementById('addWebsiteBtn').addEventListener('click', function() {
        openWebsiteModal();
    });

    document.getElementById('addHostBtn').addEventListener('click', function() {
        openHostModal();
    });

    // Close modals on outside click
    window.addEventListener('click', function(e) {
        if (e.target.classList.contains('modal')) {
            e.target.classList.remove('show');
        }
    });
}

// Navigation
function navigateToPage(page) {
    // Update sidebar
    sidebarItems.forEach(item => {
        item.classList.remove('active');
        if (item.dataset.page === page) {
            item.classList.add('active');
        }
    });

    // Update content
    pageContents.forEach(content => {
        content.classList.add('hidden');
    });

    const targetPage = document.getElementById(`${page}-page`);
    if (targetPage) {
        targetPage.classList.remove('hidden');
    }

    currentPage = page;
    loadCurrentPage();

    // Close mobile menu
    sidebar.classList.remove('show');
}

function loadCurrentPage() {
    switch (currentPage) {
        case 'dashboard':
            loadDashboard();
            break;
        case 'websites':
            loadWebsites();
            break;
        case 'hosts':
            loadHosts();
            break;
        case 'ip-management':
            loadIpManagement();
            break;
    }
}

// API Functions
async function apiRequest(endpoint, options = {}) {
    try {
        const response = await fetch(`${API_BASE}/${endpoint}`, {
            headers: {
                'Content-Type': 'application/json',
                ...options.headers
            },
            ...options
        });

        const data = await response.json();
        
        if (!data.success) {
            throw new Error(data.message || 'API request failed');
        }

        return data.data;
    } catch (error) {
        console.error('API Error:', error);
        showToast(error.message, 'error');
        throw error;
    }
}

// Dashboard Functions
async function loadDashboard() {
    try {
        showLoading(true);
        
        // Load stats
        const stats = await apiRequest('dashboard.php?stats=1');
        updateDashboardStats(stats);

        // Load IP distribution
        const ipDistribution = await apiRequest('dashboard.php?ip_distribution=1');
        displayIpDistribution(ipDistribution);

        // Load recent activity
        const recentActivity = await apiRequest('dashboard.php?recent_activity=1');
        displayRecentActivity(recentActivity);

    } catch (error) {
        console.error('Failed to load dashboard:', error);
    } finally {
        showLoading(false);
    }
}

function updateDashboardStats(stats) {
    document.getElementById('totalWebsites').textContent = stats.total_websites || 0;
    document.getElementById('activeWebsites').textContent = stats.active_websites || 0;
    document.getElementById('totalHosts').textContent = stats.total_hosts || 0;
    document.getElementById('uniqueIps').textContent = stats.unique_ips || 0;
}

function displayIpDistribution(ipDistribution) {
    const container = document.getElementById('ipDistribution');
    
    if (ipDistribution.length === 0) {
        container.innerHTML = '<p class="text-gray-500 text-center py-8">هیچ IP یافت نشد</p>';
        return;
    }

    container.innerHTML = ipDistribution.map(ip => `
        <div class="ip-card cursor-pointer" onclick="viewIpDetails('${ip.ip_address}')">
            <div class="flex justify-between items-center">
                <div>
                    <h4 class="font-bold text-lg">${ip.ip_address}</h4>
                    <div class="flex space-x-reverse space-x-4 mt-2 text-sm text-gray-600">
                        <span><i class="fas fa-globe ml-1"></i>${ip.total_websites} سایت</span>
                        <span><i class="fas fa-server ml-1"></i>${ip.total_hosts} هاست</span>
                    </div>
                </div>
                <div class="text-left">
                    <div class="text-2xl font-bold text-blue-600">${ip.total_items}</div>
                    <div class="text-xs text-gray-500">مجموع</div>
                </div>
            </div>
            <div class="mt-3">
                <div class="w-full bg-gray-200 rounded-full h-2">
                    <div class="bg-gradient-to-r from-blue-500 to-purple-600 h-2 rounded-full" 
                         style="width: ${(ip.total_items / Math.max(...ipDistribution.map(item => item.total_items))) * 100}%"></div>
                </div>
            </div>
        </div>
    `).join('');
}

function displayRecentActivity(activity) {
    const container = document.getElementById('recentActivity');
    
    if (activity.length === 0) {
        container.innerHTML = '<p class="text-gray-500 text-center py-8">هیچ فعالیتی یافت نشد</p>';
        return;
    }

    container.innerHTML = activity.map(item => `
        <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors">
            <div class="flex items-center">
                <i class="fas ${item.type === 'website' ? 'fa-globe text-blue-500' : 'fa-server text-green-500'} ml-3"></i>
                <div>
                    <div class="font-semibold">${item.name}</div>
                    <div class="text-sm text-gray-600">${item.ip_address}</div>
                </div>
            </div>
            <div class="text-left">
                <span class="status-badge status-${item.status}">${getStatusText(item.status)}</span>
                <div class="text-xs text-gray-500 mt-1">${formatDate(item.created_at)}</div>
            </div>
        </div>
    `).join('');
}

// Websites Functions
async function loadWebsites() {
    try {
        showLoading(true);
        const websites = await apiRequest('websites.php');
        displayWebsites(websites);
    } catch (error) {
        console.error('Failed to load websites:', error);
    } finally {
        showLoading(false);
    }
}

function displayWebsites(websites) {
    const container = document.getElementById('websitesList');
    
    if (websites.length === 0) {
        container.innerHTML = '<p class="text-gray-500 text-center py-8">هیچ سایتی یافت نشد</p>';
        return;
    }

    container.innerHTML = websites.map(website => `
        <div class="ip-card">
            <div class="flex justify-between items-start">
                <div class="flex-1">
                    <h4 class="font-bold text-lg">${website.name}</h4>
                    <p class="text-gray-600">${website.domain}</p>
                    <div class="flex items-center mt-2 text-sm text-gray-500">
                        <i class="fas fa-network-wired ml-2"></i>
                        <span>${website.ip_address}</span>
                        <span class="mx-2">•</span>
                        <span>${formatDate(website.created_at)}</span>
                    </div>
                    ${website.description ? `<p class="text-gray-600 mt-2">${website.description}</p>` : ''}
                </div>
                <div class="flex items-center space-x-reverse space-x-2">
                    <span class="status-badge status-${website.status}">${getStatusText(website.status)}</span>
                    <button class="btn btn-sm btn-primary" onclick="editWebsite(${website.id})">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn btn-sm btn-danger" onclick="deleteWebsite(${website.id})">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>
        </div>
    `).join('');
}

function openWebsiteModal(website = null) {
    const modal = document.getElementById('websiteModal');
    const title = document.getElementById('websiteModalTitle');
    
    if (website) {
        title.textContent = 'ویرایش سایت';
        document.getElementById('websiteId').value = website.id;
        document.getElementById('websiteName').value = website.name;
        document.getElementById('websiteDomain').value = website.domain;
        document.getElementById('websiteIp').value = website.ip_address;
        document.getElementById('websiteDescription').value = website.description || '';
        document.getElementById('websiteStatus').value = website.status;
    } else {
        title.textContent = 'افزودن سایت جدید';
        document.getElementById('websiteForm').reset();
        document.getElementById('websiteId').value = '';
    }
    
    modal.classList.add('show');
}

async function saveWebsite() {
    try {
        const id = document.getElementById('websiteId').value;
        const data = {
            name: document.getElementById('websiteName').value,
            domain: document.getElementById('websiteDomain').value,
            ip_address: document.getElementById('websiteIp').value,
            description: document.getElementById('websiteDescription').value,
            status: document.getElementById('websiteStatus').value
        };

        showLoading(true);
        
        if (id) {
            await apiRequest(`websites.php?id=${id}`, {
                method: 'PUT',
                body: JSON.stringify(data)
            });
            showToast('سایت با موفقیت ویرایش شد', 'success');
        } else {
            await apiRequest('websites.php', {
                method: 'POST',
                body: JSON.stringify(data)
            });
            showToast('سایت با موفقیت افزوده شد', 'success');
        }

        closeModal('websiteModal');
        loadWebsites();
    } catch (error) {
        console.error('Failed to save website:', error);
    } finally {
        showLoading(false);
    }
}

async function editWebsite(id) {
    try {
        const website = await apiRequest(`websites.php?id=${id}`);
        openWebsiteModal(website);
    } catch (error) {
        console.error('Failed to load website:', error);
    }
}

async function deleteWebsite(id) {
    if (!confirm('آیا از حذف این سایت اطمینان دارید؟')) {
        return;
    }

    try {
        showLoading(true);
        await apiRequest(`websites.php?id=${id}`, { method: 'DELETE' });
        showToast('سایت با موفقیت حذف شد', 'success');
        loadWebsites();
    } catch (error) {
        console.error('Failed to delete website:', error);
    } finally {
        showLoading(false);
    }
}

// Hosts Functions
async function loadHosts() {
    try {
        showLoading(true);
        const hosts = await apiRequest('hosts.php');
        displayHosts(hosts);
    } catch (error) {
        console.error('Failed to load hosts:', error);
    } finally {
        showLoading(false);
    }
}

function displayHosts(hosts) {
    const container = document.getElementById('hostsList');
    
    if (hosts.length === 0) {
        container.innerHTML = '<p class="text-gray-500 text-center py-8">هیچ هاستی یافت نشد</p>';
        return;
    }

    container.innerHTML = hosts.map(host => `
        <div class="ip-card">
            <div class="flex justify-between items-start">
                <div class="flex-1">
                    <h4 class="font-bold text-lg">${host.name}</h4>
                    <div class="flex items-center mt-2 text-sm text-gray-500">
                        <i class="fas fa-network-wired ml-2"></i>
                        <span>${host.ip_address}</span>
                        ${host.provider ? `<span class="mx-2">•</span><span>${host.provider}</span>` : ''}
                        ${host.location ? `<span class="mx-2">•</span><span>${host.location}</span>` : ''}
                    </div>
                    <div class="text-xs text-gray-500 mt-1">${formatDate(host.created_at)}</div>
                    ${host.specs ? `<p class="text-gray-600 mt-2 text-sm">${host.specs}</p>` : ''}
                </div>
                <div class="flex items-center space-x-reverse space-x-2">
                    <span class="status-badge status-${host.status}">${getStatusText(host.status)}</span>
                    <button class="btn btn-sm btn-primary" onclick="editHost(${host.id})">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn btn-sm btn-danger" onclick="deleteHost(${host.id})">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>
        </div>
    `).join('');
}

function openHostModal(host = null) {
    const modal = document.getElementById('hostModal');
    const title = document.getElementById('hostModalTitle');
    
    if (host) {
        title.textContent = 'ویرایش هاست';
        document.getElementById('hostId').value = host.id;
        document.getElementById('hostName').value = host.name;
        document.getElementById('hostIp').value = host.ip_address;
        document.getElementById('hostProvider').value = host.provider || '';
        document.getElementById('hostLocation').value = host.location || '';
        document.getElementById('hostSpecs').value = host.specs || '';
        document.getElementById('hostStatus').value = host.status;
    } else {
        title.textContent = 'افزودن هاست جدید';
        document.getElementById('hostForm').reset();
        document.getElementById('hostId').value = '';
    }
    
    modal.classList.add('show');
}

async function saveHost() {
    try {
        const id = document.getElementById('hostId').value;
        const data = {
            name: document.getElementById('hostName').value,
            ip_address: document.getElementById('hostIp').value,
            provider: document.getElementById('hostProvider').value,
            location: document.getElementById('hostLocation').value,
            specs: document.getElementById('hostSpecs').value,
            status: document.getElementById('hostStatus').value
        };

        showLoading(true);
        
        if (id) {
            await apiRequest(`hosts.php?id=${id}`, {
                method: 'PUT',
                body: JSON.stringify(data)
            });
            showToast('هاست با موفقیت ویرایش شد', 'success');
        } else {
            await apiRequest('hosts.php', {
                method: 'POST',
                body: JSON.stringify(data)
            });
            showToast('هاست با موفقیت افزوده شد', 'success');
        }

        closeModal('hostModal');
        loadHosts();
    } catch (error) {
        console.error('Failed to save host:', error);
    } finally {
        showLoading(false);
    }
}

async function editHost(id) {
    try {
        const host = await apiRequest(`hosts.php?id=${id}`);
        openHostModal(host);
    } catch (error) {
        console.error('Failed to load host:', error);
    }
}

async function deleteHost(id) {
    if (!confirm('آیا از حذف این هاست اطمینان دارید؟')) {
        return;
    }

    try {
        showLoading(true);
        await apiRequest(`hosts.php?id=${id}`, { method: 'DELETE' });
        showToast('هاست با موفقیت حذف شد', 'success');
        loadHosts();
    } catch (error) {
        console.error('Failed to delete host:', error);
    } finally {
        showLoading(false);
    }
}

// IP Management Functions
async function loadIpManagement() {
    try {
        showLoading(true);
        const ipData = await apiRequest('dashboard.php?items_by_ip=1');
        displayIpManagement(ipData);
    } catch (error) {
        console.error('Failed to load IP management:', error);
    } finally {
        showLoading(false);
    }
}

function displayIpManagement(ipData) {
    const container = document.getElementById('ipManagementList');
    
    if (ipData.length === 0) {
        container.innerHTML = '<p class="text-gray-500 text-center py-8">هیچ IP یافت نشد</p>';
        return;
    }

    container.innerHTML = ipData.map(ip => `
        <div class="ip-card">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-xl font-bold">${ip.ip_address}</h3>
                <div class="flex items-center space-x-reverse space-x-4">
                    <span class="text-sm text-gray-600">
                        <i class="fas fa-globe ml-1"></i>${ip.total_websites} سایت
                    </span>
                    <span class="text-sm text-gray-600">
                        <i class="fas fa-server ml-1"></i>${ip.total_hosts} هاست
                    </span>
                    <span class="text-lg font-bold text-blue-600">${ip.total_items} مجموع</span>
                </div>
            </div>
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <h4 class="font-semibold text-green-600 mb-2">
                        <i class="fas fa-globe ml-1"></i>سایت‌ها
                    </h4>
                    <div class="space-y-2">
                        ${ip.websites && ip.websites.length > 0 ? 
                            ip.websites.map(site => `
                                <div class="flex justify-between items-center p-2 bg-green-50 rounded">
                                    <div>
                                        <div class="font-medium">${site.name}</div>
                                        <div class="text-sm text-gray-600">${site.domain}</div>
                                    </div>
                                    <span class="status-badge status-${site.status}">${getStatusText(site.status)}</span>
                                </div>
                            `).join('') : 
                            '<p class="text-gray-500 text-sm">سایتی وجود ندارد</p>'
                        }
                    </div>
                </div>
                <div>
                    <h4 class="font-semibold text-blue-600 mb-2">
                        <i class="fas fa-server ml-1"></i>هاست‌ها
                    </h4>
                    <div class="space-y-2">
                        ${ip.hosts && ip.hosts.length > 0 ? 
                            ip.hosts.map(host => `
                                <div class="flex justify-between items-center p-2 bg-blue-50 rounded">
                                    <div>
                                        <div class="font-medium">${host.name}</div>
                                        <div class="text-sm text-gray-600">${host.provider || 'نامشخص'}</div>
                                    </div>
                                    <span class="status-badge status-${host.status}">${getStatusText(host.status)}</span>
                                </div>
                            `).join('') : 
                            '<p class="text-gray-500 text-sm">هاستی وجود ندارد</p>'
                        }
                    </div>
                </div>
            </div>
        </div>
    `).join('');
}

async function viewIpDetails(ip) {
    navigateToPage('ip-management');
    try {
        const ipDetails = await apiRequest(`dashboard.php?ip=${ip}`);
        displayIpManagement([ipDetails]);
    } catch (error) {
        console.error('Failed to load IP details:', error);
    }
}

// Search Functions
async function performSearch(query) {
    try {
        showLoading(true);
        const results = await apiRequest(`dashboard.php?search=${encodeURIComponent(query)}`);
        displaySearchResults(results);
    } catch (error) {
        console.error('Search failed:', error);
    } finally {
        showLoading(false);
    }
}

function displaySearchResults(results) {
    // Hide all pages and show search results
    pageContents.forEach(content => {
        content.classList.add('hidden');
    });

    // Create search results page if it doesn't exist
    let searchPage = document.getElementById('search-results-page');
    if (!searchPage) {
        searchPage = document.createElement('div');
        searchPage.id = 'search-results-page';
        searchPage.className = 'page-content';
        document.querySelector('.main-content').appendChild(searchPage);
    }

    searchPage.classList.remove('hidden');
    
    let html = `
        <div class="glass-morphism rounded-lg p-6">
            <h2 class="text-2xl font-bold text-gray-800 mb-6">
                <i class="fas fa-search ml-2"></i>
                نتایج جستجو برای "${globalSearch.value}"
            </h2>
    `;

    if (results.websites && results.websites.length > 0) {
        html += `
            <div class="mb-8">
                <h3 class="text-lg font-semibold text-green-600 mb-4">
                    <i class="fas fa-globe ml-2"></i>سایت‌ها (${results.websites.length})
                </h3>
                <div class="space-y-3">
                    ${results.websites.map(site => `
                        <div class="ip-card">
                            <div class="flex justify-between items-center">
                                <div>
                                    <h4 class="font-bold">${site.name}</h4>
                                    <p class="text-gray-600">${site.domain}</p>
                                    <div class="text-sm text-gray-500 mt-1">
                                        <i class="fas fa-network-wired ml-1"></i>${site.ip_address}
                                    </div>
                                </div>
                                <span class="status-badge status-${site.status}">${getStatusText(site.status)}</span>
                            </div>
                        </div>
                    `).join('')}
                </div>
            </div>
        `;
    }

    if (results.hosts && results.hosts.length > 0) {
        html += `
            <div class="mb-8">
                <h3 class="text-lg font-semibold text-blue-600 mb-4">
                    <i class="fas fa-server ml-2"></i>هاست‌ها (${results.hosts.length})
                </h3>
                <div class="space-y-3">
                    ${results.hosts.map(host => `
                        <div class="ip-card">
                            <div class="flex justify-between items-center">
                                <div>
                                    <h4 class="font-bold">${host.name}</h4>
                                    <div class="text-sm text-gray-500">
                                        <i class="fas fa-network-wired ml-1"></i>${host.ip_address}
                                        ${host.provider ? `<span class="mx-2">•</span>${host.provider}` : ''}
                                    </div>
                                </div>
                                <span class="status-badge status-${host.status}">${getStatusText(host.status)}</span>
                            </div>
                        </div>
                    `).join('')}
                </div>
            </div>
        `;
    }

    if ((!results.websites || results.websites.length === 0) && 
        (!results.hosts || results.hosts.length === 0)) {
        html += '<p class="text-gray-500 text-center py-8">هیچ نتیجه‌ای یافت نشد</p>';
    }

    html += '</div>';
    searchPage.innerHTML = html;
}

// Utility Functions
function closeModal(modalId) {
    document.getElementById(modalId).classList.remove('show');
}

function showToast(message, type = 'success') {
    const toast = document.createElement('div');
    toast.className = `toast ${type}`;
    toast.textContent = message;
    document.body.appendChild(toast);

    setTimeout(() => {
        toast.remove();
    }, 3000);
}

function showLoading(show) {
    isLoading = show;
    // You can add a global loading indicator here
}

function getStatusText(status) {
    const statusMap = {
        'active': 'فعال',
        'inactive': 'غیرفعال',
        'suspended': 'مسدود',
        'maintenance': 'در حال تعمیر'
    };
    return statusMap[status] || status;
}

function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('fa-IR', {
        year: 'numeric',
        month: 'long',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Add CSS for button-sm class
const style = document.createElement('style');
style.textContent = `
    .btn-sm {
        padding: 6px 12px;
        font-size: 12px;
    }
    .btn-secondary {
        background: #6b7280;
        color: white;
    }
    .btn-secondary:hover {
        background: #4b5563;
    }
`;
document.head.appendChild(style);