<?php
require_once '../config/database.php';

class WebsiteAPI {
    private $db;

    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
    }

    // GET all websites
    public function getAll() {
        $stmt = $this->db->prepare("
            SELECT w.*, 
                   COUNT(wa.id) as assignment_count
            FROM websites w
            LEFT JOIN ip_assignments wa ON w.id = wa.website_id
            GROUP BY w.id
            ORDER BY w.created_at DESC
        ");
        $stmt->execute();
        return $stmt->fetchAll();
    }

    // GET website by ID
    public function getById($id) {
        $stmt = $this->db->prepare("SELECT * FROM websites WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    // GET websites by IP
    public function getByIp($ip) {
        $stmt = $this->db->prepare("SELECT * FROM websites WHERE ip_address = ? ORDER BY created_at DESC");
        $stmt->execute([$ip]);
        return $stmt->fetchAll();
    }

    // CREATE new website
    public function create($data) {
        $stmt = $this->db->prepare("
            INSERT INTO websites (name, domain, ip_address, description, status) 
            VALUES (?, ?, ?, ?, ?)
        ");
        return $stmt->execute([
            $data['name'],
            $data['domain'],
            $data['ip_address'],
            $data['description'] ?? null,
            $data['status'] ?? 'active'
        ]);
    }

    // UPDATE website
    public function update($id, $data) {
        $stmt = $this->db->prepare("
            UPDATE websites 
            SET name = ?, domain = ?, ip_address = ?, description = ?, status = ?
            WHERE id = ?
        ");
        return $stmt->execute([
            $data['name'],
            $data['domain'],
            $data['ip_address'],
            $data['description'] ?? null,
            $data['status'] ?? 'active',
            $id
        ]);
    }

    // DELETE website
    public function delete($id) {
        $stmt = $this->db->prepare("DELETE FROM websites WHERE id = ?");
        return $stmt->execute([$id]);
    }

    // GET unique IPs from websites
    public function getUniqueIps() {
        $stmt = $this->db->prepare("
            SELECT ip_address, COUNT(*) as count 
            FROM websites 
            GROUP BY ip_address 
            ORDER BY count DESC
        ");
        $stmt->execute();
        return $stmt->fetchAll();
    }
}

// Handle requests
$websiteAPI = new WebsiteAPI();
$method = $_SERVER['REQUEST_METHOD'];
$input = json_decode(file_get_contents('php://input'), true);

try {
    switch ($method) {
        case 'GET':
            if (isset($_GET['id'])) {
                $result = $websiteAPI->getById($_GET['id']);
            } elseif (isset($_GET['ip'])) {
                $result = $websiteAPI->getByIp($_GET['ip']);
            } elseif (isset($_GET['ips'])) {
                $result = $websiteAPI->getUniqueIps();
            } else {
                $result = $websiteAPI->getAll();
            }
            ApiResponse::success($result);
            break;

        case 'POST':
            if (!$input || !isset($input['name']) || !isset($input['domain']) || !isset($input['ip_address'])) {
                ApiResponse::error('Missing required fields: name, domain, ip_address');
            }
            
            if ($websiteAPI->create($input)) {
                ApiResponse::success(null, 'Website created successfully');
            } else {
                ApiResponse::error('Failed to create website');
            }
            break;

        case 'PUT':
            if (!isset($_GET['id'])) {
                ApiResponse::error('Website ID is required');
            }
            
            if ($websiteAPI->update($_GET['id'], $input)) {
                ApiResponse::success(null, 'Website updated successfully');
            } else {
                ApiResponse::error('Failed to update website');
            }
            break;

        case 'DELETE':
            if (!isset($_GET['id'])) {
                ApiResponse::error('Website ID is required');
            }
            
            if ($websiteAPI->delete($_GET['id'])) {
                ApiResponse::success(null, 'Website deleted successfully');
            } else {
                ApiResponse::error('Failed to delete website');
            }
            break;

        default:
            ApiResponse::error('Method not allowed', 405);
    }
} catch (Exception $e) {
    ApiResponse::error($e->getMessage());
}
?>