<?php
require_once '../config/database.php';

class HostAPI {
    private $db;

    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
    }

    // GET all hosts
    public function getAll() {
        $stmt = $this->db->prepare("
            SELECT h.*, 
                   COUNT(wa.id) as assignment_count
            FROM hosts h
            LEFT JOIN ip_assignments wa ON h.id = wa.host_id
            GROUP BY h.id
            ORDER BY h.created_at DESC
        ");
        $stmt->execute();
        return $stmt->fetchAll();
    }

    // GET host by ID
    public function getById($id) {
        $stmt = $this->db->prepare("SELECT * FROM hosts WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    // GET hosts by IP
    public function getByIp($ip) {
        $stmt = $this->db->prepare("SELECT * FROM hosts WHERE ip_address = ? ORDER BY created_at DESC");
        $stmt->execute([$ip]);
        return $stmt->fetchAll();
    }

    // CREATE new host
    public function create($data) {
        $stmt = $this->db->prepare("
            INSERT INTO hosts (name, ip_address, provider, location, specs, status) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        return $stmt->execute([
            $data['name'],
            $data['ip_address'],
            $data['provider'] ?? null,
            $data['location'] ?? null,
            $data['specs'] ?? null,
            $data['status'] ?? 'active'
        ]);
    }

    // UPDATE host
    public function update($id, $data) {
        $stmt = $this->db->prepare("
            UPDATE hosts 
            SET name = ?, ip_address = ?, provider = ?, location = ?, specs = ?, status = ?
            WHERE id = ?
        ");
        return $stmt->execute([
            $data['name'],
            $data['ip_address'],
            $data['provider'] ?? null,
            $data['location'] ?? null,
            $data['specs'] ?? null,
            $data['status'] ?? 'active',
            $id
        ]);
    }

    // DELETE host
    public function delete($id) {
        $stmt = $this->db->prepare("DELETE FROM hosts WHERE id = ?");
        return $stmt->execute([$id]);
    }

    // GET unique IPs from hosts
    public function getUniqueIps() {
        $stmt = $this->db->prepare("
            SELECT ip_address, COUNT(*) as count 
            FROM hosts 
            GROUP BY ip_address 
            ORDER BY count DESC
        ");
        $stmt->execute();
        return $stmt->fetchAll();
    }
}

// Handle requests
$hostAPI = new HostAPI();
$method = $_SERVER['REQUEST_METHOD'];
$input = json_decode(file_get_contents('php://input'), true);

try {
    switch ($method) {
        case 'GET':
            if (isset($_GET['id'])) {
                $result = $hostAPI->getById($_GET['id']);
            } elseif (isset($_GET['ip'])) {
                $result = $hostAPI->getByIp($_GET['ip']);
            } elseif (isset($_GET['ips'])) {
                $result = $hostAPI->getUniqueIps();
            } else {
                $result = $hostAPI->getAll();
            }
            ApiResponse::success($result);
            break;

        case 'POST':
            if (!$input || !isset($input['name']) || !isset($input['ip_address'])) {
                ApiResponse::error('Missing required fields: name, ip_address');
            }
            
            if ($hostAPI->create($input)) {
                ApiResponse::success(null, 'Host created successfully');
            } else {
                ApiResponse::error('Failed to create host');
            }
            break;

        case 'PUT':
            if (!isset($_GET['id'])) {
                ApiResponse::error('Host ID is required');
            }
            
            if ($hostAPI->update($_GET['id'], $input)) {
                ApiResponse::success(null, 'Host updated successfully');
            } else {
                ApiResponse::error('Failed to update host');
            }
            break;

        case 'DELETE':
            if (!isset($_GET['id'])) {
                ApiResponse::error('Host ID is required');
            }
            
            if ($hostAPI->delete($_GET['id'])) {
                ApiResponse::success(null, 'Host deleted successfully');
            } else {
                ApiResponse::error('Failed to delete host');
            }
            break;

        default:
            ApiResponse::error('Method not allowed', 405);
    }
} catch (Exception $e) {
    ApiResponse::error($e->getMessage());
}
?>