<?php
require_once '../config/database.php';

class DashboardAPI {
    private $db;

    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
    }

    // GET dashboard statistics
    public function getStats() {
        $stats = [];

        // Total websites
        $stmt = $this->db->prepare("SELECT COUNT(*) as total FROM websites");
        $stmt->execute();
        $stats['total_websites'] = $stmt->fetch()['total'];

        // Active websites
        $stmt = $this->db->prepare("SELECT COUNT(*) as total FROM websites WHERE status = 'active'");
        $stmt->execute();
        $stats['active_websites'] = $stmt->fetch()['total'];

        // Total hosts
        $stmt = $this->db->prepare("SELECT COUNT(*) as total FROM hosts");
        $stmt->execute();
        $stats['total_hosts'] = $stmt->fetch()['total'];

        // Active hosts
        $stmt = $this->db->prepare("SELECT COUNT(*) as total FROM hosts WHERE status = 'active'");
        $stmt->execute();
        $stats['active_hosts'] = $stmt->fetch()['total'];

        // Unique IPs
        $stmt = $this->db->prepare("
            SELECT COUNT(DISTINCT ip_address) as total FROM (
                SELECT ip_address FROM websites 
                UNION 
                SELECT ip_address FROM hosts
            ) as all_ips
        ");
        $stmt->execute();
        $stats['unique_ips'] = $stmt->fetch()['total'];

        return $stats;
    }

    // GET IP distribution
    public function getIpDistribution() {
        $stmt = $this->db->prepare("
            SELECT 
                ip_address,
                SUM(website_count) as total_websites,
                SUM(host_count) as total_hosts,
                SUM(website_count + host_count) as total_items
            FROM (
                SELECT 
                    ip_address,
                    COUNT(*) as website_count,
                    0 as host_count
                FROM websites
                GROUP BY ip_address
                UNION ALL
                SELECT 
                    ip_address,
                    0 as website_count,
                    COUNT(*) as host_count
                FROM hosts
                GROUP BY ip_address
            ) as combined
            GROUP BY ip_address
            ORDER BY total_items DESC
        ");
        $stmt->execute();
        return $stmt->fetchAll();
    }

    // GET recent activity
    public function getRecentActivity() {
        $stmt = $this->db->prepare("
            (SELECT 'website' as type, name, ip_address, created_at, status FROM websites ORDER BY created_at DESC LIMIT 5)
            UNION ALL
            (SELECT 'host' as type, name, ip_address, created_at, status FROM hosts ORDER BY created_at DESC LIMIT 5)
            ORDER BY created_at DESC
            LIMIT 10
        ");
        $stmt->execute();
        return $stmt->fetchAll();
    }

    // GET items by IP with details
    public function getItemsByIp($ip = null) {
        if ($ip) {
            // Get specific IP details
            $websites = [];
            $hosts = [];

            $stmt = $this->db->prepare("SELECT * FROM websites WHERE ip_address = ?");
            $stmt->execute([$ip]);
            $websites = $stmt->fetchAll();

            $stmt = $this->db->prepare("SELECT * FROM hosts WHERE ip_address = ?");
            $stmt->execute([$ip]);
            $hosts = $stmt->fetchAll();

            return [
                'ip_address' => $ip,
                'websites' => $websites,
                'hosts' => $hosts,
                'total_items' => count($websites) + count($hosts)
            ];
        } else {
            // Get all IPs with summary
            return $this->getIpDistribution();
        }
    }

    // Search functionality
    public function search($query) {
        $results = [];
        
        // Search websites
        $stmt = $this->db->prepare("
            SELECT 'website' as type, id, name, domain, ip_address, description, status, created_at 
            FROM websites 
            WHERE name LIKE ? OR domain LIKE ? OR ip_address LIKE ? OR description LIKE ?
        ");
        $searchTerm = "%{$query}%";
        $stmt->execute([$searchTerm, $searchTerm, $searchTerm, $searchTerm]);
        $results['websites'] = $stmt->fetchAll();

        // Search hosts
        $stmt = $this->db->prepare("
            SELECT 'host' as type, id, name, ip_address, provider, location, specs, status, created_at 
            FROM hosts 
            WHERE name LIKE ? OR ip_address LIKE ? OR provider LIKE ? OR location LIKE ? OR specs LIKE ?
        ");
        $stmt->execute([$searchTerm, $searchTerm, $searchTerm, $searchTerm, $searchTerm]);
        $results['hosts'] = $stmt->fetchAll();

        return $results;
    }
}

// Handle requests
$dashboardAPI = new DashboardAPI();
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            if (isset($_GET['stats'])) {
                $result = $dashboardAPI->getStats();
            } elseif (isset($_GET['ip_distribution'])) {
                $result = $dashboardAPI->getIpDistribution();
            } elseif (isset($_GET['recent_activity'])) {
                $result = $dashboardAPI->getRecentActivity();
            } elseif (isset($_GET['ip'])) {
                $result = $dashboardAPI->getItemsByIp($_GET['ip']);
            } elseif (isset($_GET['items_by_ip'])) {
                $result = $dashboardAPI->getItemsByIp();
            } elseif (isset($_GET['search'])) {
                $result = $dashboardAPI->search($_GET['search']);
            } else {
                // Default to stats
                $result = $dashboardAPI->getStats();
            }
            ApiResponse::success($result);
            break;

        default:
            ApiResponse::error('Method not allowed', 405);
    }
} catch (Exception $e) {
    ApiResponse::error($e->getMessage());
}
?>